<?php
require_once('source/session.php');
require_once('source/error.php');
require_once('source/dbconfig.php');
require_once('source/function.php');
session_start();

// Check if the user is logged in
if (isset($_SESSION['SESS_ID_ADMIN'])) {
    $userId = $_SESSION['SESS_ID_ADMIN'];

    // Fetch user details based on the ID
    $query = "SELECT * FROM tbl_admin WHERE id = :id";
    $stmt = $DBH->prepare($query);
    $stmt->bindParam(':id', $userId, PDO::PARAM_INT);
    $stmt->execute();

    if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $adminName = $row['username'];
    }
}

// Fetch dynamic options for class, term, and year from the database
$classes = sql($DBH, "SELECT DISTINCT class FROM tbl_student_results ORDER BY class", [], "rows");
$terms = sql($DBH, "SELECT DISTINCT term FROM tbl_student_results ORDER BY term", [], "rows");
$years = sql($DBH, "SELECT DISTINCT year FROM tbl_student_results ORDER BY year DESC", [], "rows");

// Initialize variables
$class = $_POST['class'] ?? '';
$term = $_POST['term'] ?? '';
$year = $_POST['year'] ?? '';
$page_title = '';
$total_results_count = 0;
$class_results = [];

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Extract search data from the form
    $class = strip_tags($_POST['class']);
    $term = strip_tags($_POST['term']);
    $year = strip_tags($_POST['year']);

    // Retrieve uploaded class results from the database based on the search criteria
    $class_results = sql($DBH, 
        "SELECT DISTINCT class, term, year, subject, csv_no, 
                COUNT(*) as student_count,
                MIN(id) as first_id
         FROM tbl_student_results 
         WHERE class = ? AND term = ? AND year = ? 
         GROUP BY class, term, year, subject, csv_no
         ORDER BY subject, csv_no",
        [$class, $term, $year], 
        "rows"
    );

    // Set page title
    $page_title = "Class: $class, Term: $term, Year: $year";

    // Count of total uploaded class results
    $total_results_count = count($class_results);
}

// Handle delete action
if (isset($_GET['delete_csv']) && !empty($_GET['delete_csv'])) {
    $csv_no = strip_tags($_GET['delete_csv']);
    
    // Get details before deletion for confirmation message
    $details = sql($DBH, 
        "SELECT class, term, year, subject FROM tbl_student_results WHERE csv_no = ? LIMIT 1", 
        [$csv_no], 
        "row"
    );
    
    // Delete records
    $delete_query = "DELETE FROM tbl_student_results WHERE csv_no = ?";
    $stmt = $DBH->prepare($delete_query);
    
    if ($stmt->execute([$csv_no])) {
        $success_message = "Successfully deleted all records with CSV Number: $csv_no";
        if ($details) {
            $success_message .= " ({$details['subject']} - {$details['class']})";
        }
        
        // Refresh the current results if they match the deleted CSV
        if (!empty($class) && !empty($term) && !empty($year)) {
            $class_results = sql($DBH, 
                "SELECT DISTINCT class, term, year, subject, csv_no, 
                        COUNT(*) as student_count,
                        MIN(id) as first_id
                 FROM tbl_student_results 
                 WHERE class = ? AND term = ? AND year = ? 
                 GROUP BY class, term, year, subject, csv_no
                 ORDER BY subject, csv_no",
                [$class, $term, $year], 
                "rows"
            );
            $total_results_count = count($class_results);
        }
    } else {
        $error_message = "Failed to delete records with CSV Number: $csv_no";
    }
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
  <title>VERBUM DEI ACADEMY - Uploaded Results</title>
  <link rel="icon" type="image/png" href="https://portal.vda.com.ng/pri_admin/pri_logo.jpg">
  
  <!-- Bootstrap 5 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  
  <!-- Font Awesome -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  
  <!-- Google Fonts - Nunito -->
  <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@400;600;700&display=swap" rel="stylesheet">
  
  <!-- Custom CSS -->
  <style>
    :root {
      --primary-color: #1976d2;
      --secondary-color: #64b5f6;
      --accent-color: #bbdefb;
      --dark-color: #0d47a1;
      --light-color: #e3f2fd;
      --text-dark: #212529;
      --text-light: #f8f9fa;
    }
    
    body {
      font-family: 'Nunito', sans-serif;
      background-color: #f5f7fa;
      padding-top: 56px;
      color: var(--text-dark);
      font-size: 0.9rem;
    }
    
    /* Sidebar Styles */
    .sidebar {
      background: linear-gradient(180deg, var(--dark-color) 0%, var(--primary-color) 100%);
      position: fixed;
      top: 56px;
      left: 0;
      width: 250px;
      height: calc(100vh - 56px);
      transition: all 0.3s;
      z-index: 1000;
      overflow-y: auto;
    }
    
    .sidebar .nav-link {
      color: rgba(255, 255, 255, 0.85);
      font-weight: 600;
      padding: 0.75rem 1rem;
      margin-bottom: 0.2rem;
      transition: all 0.2s;
    }
    
    .sidebar .nav-link:hover {
      color: #fff;
      background-color: rgba(255, 255, 255, 0.15);
    }
    
    .sidebar .nav-link.active {
      color: #fff;
      background-color: var(--secondary-color);
      box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    
    .sidebar .nav-link i {
      margin-right: 0.5rem;
      width: 20px;
      text-align: center;
    }
    
    /* Main Content Styles */
    .main-content {
      margin-left: 250px;
      width: calc(100% - 250px);
      transition: all 0.3s;
      min-height: calc(100vh - 56px);
      padding-bottom: 60px;
    }
    
    @media (max-width: 991.98px) {
      .sidebar {
        left: -250px;
      }
      .main-content {
        margin-left: 0;
        width: 100%;
      }
      body {
        padding-top: 56px;
      }
    }
    
    /* Navbar */
    .navbar {
      background-color: var(--primary-color);
      box-shadow: 0 0.15rem 1.75rem 0 rgba(33, 37, 41, 0.1);
      position: fixed;
      top: 0;
      width: 100%;
      z-index: 1030;
      color: white;
    }
    
    /* Welcome banner */
    .welcome-banner {
      background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
      color: white;
      border-radius: 0.5rem;
      padding: 0.75rem 1rem;
      margin: 1rem;
      box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    
    /* Form Styles */
    .form-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
    }
    
    /* Table Styles */
    .table-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
      overflow-x: auto;
    }
    
    .results-table {
      width: 100%;
      border-collapse: collapse;
      font-size: 0.85rem;
    }
    
    .results-table th,
    .results-table td {
      padding: 0.75rem;
      border: 1px solid #dee2e6;
      text-align: left;
      vertical-align: middle;
    }
    
    .results-table th {
      background-color: var(--primary-color);
      color: white;
      font-weight: 600;
      position: sticky;
      top: 0;
      z-index: 10;
    }
    
    .results-table tr:nth-child(even) {
      background-color: #f9f9f9;
    }
    
    .results-table tr:hover {
      background-color: #f1f1f1;
    }
    
    /* Badge styles */
    .badge-count {
      background-color: var(--secondary-color);
      color: white;
      font-size: 0.75rem;
      padding: 0.2rem 0.5rem;
      border-radius: 0.25rem;
    }
    
    /* Action buttons */
    .action-buttons {
      display: flex;
      gap: 0.5rem;
    }
    
    .btn-sm {
      padding: 0.25rem 0.5rem;
      font-size: 0.8rem;
      border-radius: 0.25rem;
    }
    
    .btn-edit {
      background-color: #28a745;
      color: white;
      border: none;
    }
    
    .btn-edit:hover {
      background-color: #218838;
    }
    
    .btn-delete {
      background-color: #dc3545;
      color: white;
      border: none;
    }
    
    .btn-delete:hover {
      background-color: #c82333;
    }
    
    .btn-view {
      background-color: #17a2b8;
      color: white;
      border: none;
    }
    
    .btn-view:hover {
      background-color: #138496;
    }
    
    /* Footer styles */
    .footer {
      background-color: var(--dark-color);
      color: white;
      padding: 12px 0;
      position: fixed;
      bottom: 0;
      width: 100%;
      z-index: 100;
      font-size: 0.8rem;
    }
    
    .copyright-text {
      font-weight: 600;
    }
    
    .powered-by {
      display: flex;
      align-items: center;
      justify-content: flex-end;
      color: white;
    }
    
    .powered-by img {
      height: 18px;
      margin-left: 6px;
    }
    
    .powered-by-link {
      display: flex;
      align-items: center;
      text-decoration: none;
      color: inherit;
      transition: opacity 0.3s ease;
    }
    
    .powered-by-link:hover {
      opacity: 0.8;
      text-decoration: none;
    }
    
    /* Action buttons in main */
    .btn-action {
      padding: 0.5rem 1rem;
      border-radius: 0.25rem;
      font-weight: 600;
      transition: all 0.3s;
      font-size: 0.85rem;
    }
    
    .btn-primary {
      background-color: var(--primary-color);
      color: white;
      border: none;
    }
    
    .btn-primary:hover {
      background-color: var(--dark-color);
    }
    
    /* Alert styles */
    .alert {
      padding: 0.75rem 1rem;
      border-radius: 0.25rem;
      margin: 1rem;
    }
    
    .alert-success {
      background-color: #d4edda;
      border-color: #c3e6cb;
      color: #155724;
    }
    
    .alert-danger {
      background-color: #f8d7da;
      border-color: #f5c6cb;
      color: #721c24;
    }
    
    /* Info cards */
    .info-card {
      background: linear-gradient(135deg, var(--light-color) 0%, #ffffff 100%);
      border-left: 4px solid var(--primary-color);
      padding: 0.75rem;
      margin-bottom: 0.5rem;
      border-radius: 0 0.25rem 0.25rem 0;
    }
    
    /* Compact form elements */
    .form-select, .form-label {
      font-size: 0.85rem;
    }
    
    /* Empty state */
    .empty-state {
      text-align: center;
      padding: 2rem;
      color: #6c757d;
    }
    
    .empty-state i {
      font-size: 3rem;
      margin-bottom: 1rem;
    }
  </style>
</head>

<body>
  <!-- Top Navigation -->
  <nav class="navbar navbar-expand-lg navbar-light">
    <div class="container-fluid">
      <button id="sidebarToggle" class="btn btn-link d-lg-none me-2">
        <i class="fas fa-bars"></i>
      </button>
      <a class="navbar-brand d-none d-lg-block" href="#">
        VERBUM DEI ACADEMY
      </a>
      <div class="ms-auto d-flex align-items-center">
        <div class="admin-info me-3">
          <span class="admin-name"><?php echo htmlspecialchars($adminName ?? 'Admin'); ?></span>
          <span class="admin-access">(Administrator)</span>
          <i class="fas fa-user-circle admin-icon"></i>
        </div>
        <a href="logout.php" class="btn btn-sm btn-outline-light">
          <i class="fas fa-sign-out-alt"></i> Logout
        </a>
      </div>
    </div>
  </nav>

  <!-- Sidebar -->
  <nav id="sidebar" class="sidebar">
    <div class="position-sticky pt-3">
      <div class="text-center mb-4 d-none d-lg-block">
        <img src="https://portal.vda.com.ng/pri_admin/pri_logo.jpg" alt="School Logo" class="rounded-circle" width="80">
        <h5 class="mt-2 text-white">VERBUM DEI ACADEMY</h5>
        <p class="text-white-50 small">Admin Dashboard</p>
      </div>
      
      <ul class="nav flex-column">
        <li class="nav-item">
          <a class="nav-link" href="main.php">
            <i class="fas fa-fw fa-tachometer-alt"></i>
            Admin Dashboard
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="edit_scores_subject.php">
            <i class="fas fa-fw fa-edit"></i>
            Edit Subject Scores
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="download_broadsheet.php">
            <i class="fas fa-fw fa-file-pdf"></i>
            Result Broadsheet
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link active" href="list_uploaded_results.php">
            <i class="fas fa-fw fa-list"></i>
            Uploaded Results
          </a>
        </li>
      </ul>
    </div>
  </nav>

  <!-- Main Content -->
  <main class="main-content">
    <!-- Welcome Banner -->
    <div class="welcome-banner">
      <div class="d-flex justify-content-between align-items-center">
        <div>
          <h4 class="mb-1">Uploaded Results Management</h4>
          <p class="mb-0 small">VERBUM DEI ACADEMY Admin Dashboard</p>
        </div>
        <div class="d-none d-sm-inline-block">
          <span class="badge bg-white text-primary p-1 small">
            <i class="fas fa-database me-1"></i> Data Management
          </span>
        </div>
      </div>
    </div>

    <!-- Search Form -->
    <div class="form-container">
      <form method="POST" action="">
        <div class="row align-items-end">
          <div class="col-md-4">
            <div class="form-group mb-2">
              <label for="class" class="form-label">Class:</label>
              <select id="class" name="class" class="form-select form-select-sm" required>
                <option value="">Select Class</option>
                <?php foreach ($classes as $classOption): ?>
                  <option value="<?php echo htmlspecialchars($classOption['class']); ?>" 
                    <?php echo ($classOption['class'] === $class) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($classOption['class']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="term" class="form-label">Term:</label>
              <select id="term" name="term" class="form-select form-select-sm" required>
                <option value="">Select Term</option>
                <?php foreach ($terms as $termOption): ?>
                  <option value="<?php echo htmlspecialchars($termOption['term']); ?>" 
                    <?php echo ($termOption['term'] === $term) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($termOption['term']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="year" class="form-label">Academic Year:</label>
              <select id="year" name="year" class="form-select form-select-sm" required>
                <option value="">Select Year</option>
                <?php foreach ($years as $yearOption): ?>
                  <option value="<?php echo htmlspecialchars($yearOption['year']); ?>" 
                    <?php echo ($yearOption['year'] === $year) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($yearOption['year']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-2">
            <button type="submit" class="btn btn-primary btn-sm w-100">
              <i class="fas fa-search me-1"></i> Search Results
            </button>
          </div>
        </div>
      </form>
    </div>

    <?php if (isset($error_message)): ?>
      <div class="alert alert-danger">
        <i class="fas fa-exclamation-circle me-2"></i> <?php echo $error_message; ?>
      </div>
    <?php endif; ?>

    <?php if (isset($success_message)): ?>
      <div class="alert alert-success">
        <i class="fas fa-check-circle me-2"></i> <?php echo $success_message; ?>
      </div>
    <?php endif; ?>

    <?php if (!empty($page_title)): ?>
      <div class="form-container">
        <div class="row">
          <div class="col-md-8">
            <div class="info-card">
              <h6 class="mb-1"><i class="fas fa-filter me-2"></i>Filter Results: <?php echo htmlspecialchars($page_title); ?></h6>
              <p class="mb-0 small">Showing uploaded subject results for the selected criteria</p>
            </div>
          </div>
          <div class="col-md-4">
            <div class="info-card">
              <h6 class="mb-1"><i class="fas fa-layer-group me-2"></i>Total Uploads: <?php echo $total_results_count; ?></h6>
              <p class="mb-0 small">Number of distinct subject uploads found</p>
            </div>
          </div>
        </div>
      </div>
    <?php endif; ?>

    <?php if (!empty($class_results)): ?>
      <div class="table-container">
        <div class="d-flex justify-content-between align-items-center mb-3">
          <h5 class="mb-0">
            Uploaded Subject Results
            <small class="text-muted">(<?php echo $total_results_count; ?> uploads)</small>
          </h5>
          <div class="btn-group" role="group">
            <button type="button" class="btn btn-outline-primary btn-sm" onclick="window.print()">
              <i class="fas fa-print me-1"></i> Print
            </button>
            <button type="button" class="btn btn-outline-success btn-sm" onclick="exportToCSV()">
              <i class="fas fa-download me-1"></i> Export
            </button>
          </div>
        </div>
        
        <div class="table-responsive">
          <table class="results-table">
            <thead>
              <tr>
                <th width="5%">S/N</th>
                <th>Subject</th>
                <th>Class</th>
                <th>Term</th>
                <th>Year</th>
                <th>CSV Number</th>
                <th>Students</th>
                <th width="15%">Actions</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($class_results as $index => $record): 
                $student_count = sql($DBH, 
                  "SELECT COUNT(*) as count FROM tbl_student_results 
                   WHERE csv_no = ? AND class = ? AND term = ? AND year = ? AND subject = ?",
                  [$record['csv_no'], $record['class'], $record['term'], $record['year'], $record['subject']], 
                  "row"
                );
                $student_count = $student_count['count'] ?? 0;
              ?>
                <tr>
                  <td><?php echo $index + 1; ?></td>
                  <td>
                    <strong><?php echo htmlspecialchars($record['subject']); ?></strong>
                  </td>
                  <td><?php echo htmlspecialchars($record['class']); ?></td>
                  <td><?php echo htmlspecialchars($record['term']); ?></td>
                  <td><?php echo htmlspecialchars($record['year']); ?></td>
                  <td>
                    <code><?php echo htmlspecialchars($record['csv_no']); ?></code>
                  </td>
                  <td>
                    <span class="badge-count"><?php echo $student_count; ?> students</span>
                  </td>
                  <td>
                    <div class="action-buttons">
                      <a href="edit_scores_subject.php?class=<?php echo urlencode($record['class']); ?>&term=<?php echo urlencode($record['term']); ?>&year=<?php echo urlencode($record['year']); ?>&subject=<?php echo urlencode($record['subject']); ?>" 
                         class="btn btn-edit btn-sm" title="Edit Scores">
                        <i class="fas fa-edit"></i>
                      </a>
                      <a href="view_subject_results.php?class=<?php echo urlencode($record['class']); ?>&term=<?php echo urlencode($record['term']); ?>&year=<?php echo urlencode($record['year']); ?>&subject=<?php echo urlencode($record['subject']); ?>" 
                         class="btn btn-view btn-sm" title="View Results" target="_blank">
                        <i class="fas fa-eye"></i>
                      </a>
                      <a href="?delete_csv=<?php echo urlencode($record['csv_no']); ?>&class=<?php echo urlencode($class); ?>&term=<?php echo urlencode($term); ?>&year=<?php echo urlencode($year); ?>" 
                         class="btn btn-delete btn-sm" 
                         onclick="return confirmDelete('<?php echo addslashes($record['csv_no']); ?>', '<?php echo addslashes($record['subject']); ?>', '<?php echo addslashes($record['class']); ?>')"
                         title="Delete All Records">
                        <i class="fas fa-trash"></i>
                      </a>
                    </div>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
        
        <div class="mt-3">
          <div class="alert alert-info py-2">
            <h6 class="mb-1"><i class="fas fa-info-circle me-1"></i> Actions Guide</h6>
            <div class="row small">
              <div class="col-md-4">
                <i class="fas fa-edit text-success me-1"></i> Edit: Modify individual scores
              </div>
              <div class="col-md-4">
                <i class="fas fa-eye text-info me-1"></i> View: See detailed results
              </div>
              <div class="col-md-4">
                <i class="fas fa-trash text-danger me-1"></i> Delete: Remove entire upload
              </div>
            </div>
          </div>
        </div>
      </div>
    <?php elseif ($_SERVER["REQUEST_METHOD"] == "POST" && empty($class_results)): ?>
      <div class="form-container text-center py-4">
        <i class="fas fa-search fa-3x mb-2 text-muted"></i>
        <h5 class="text-muted">No Uploads Found</h5>
        <p class="mb-0">There are no uploaded results for the selected criteria.</p>
        <div class="mt-3">
          <a href="upload_results.php" class="btn btn-primary btn-sm">
            <i class="fas fa-upload me-1"></i> Upload New Results
          </a>
        </div>
      </div>
    <?php else: ?>
      <div class="form-container text-center py-4">
        <i class="fas fa-database fa-3x mb-2 text-muted"></i>
        <h5 class="text-muted">Uploaded Results Database</h5>
        <p class="mb-0">Select a class, term, and academic year to view uploaded results.</p>
      </div>
    <?php endif; ?>
  </main>

  <!-- Footer -->
  <footer class="footer">
    <div class="container-fluid">
      <div class="row align-items-center">
        <div class="col-md-6">
          <span class="copyright-text">Copyright &copy; VERBUM DEI ACADEMY <?php echo date("Y"); ?></span>
        </div>
        <div class="col-md-6">
          <div class="powered-by">
            <a href="https://dataswift.com.ng" target="_blank" rel="noopener noreferrer" class="powered-by-link">
              <span>Powered by DataSwift</span>
              <img src="datalogo.png" alt="DataSwift Logo">
            </a>
          </div>
        </div>
      </div>
    </div>
  </footer>

  <!-- Bootstrap 5 JS Bundle with Popper -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
  
  <!-- Custom Scripts -->
  <script>
    // Toggle sidebar on mobile
    document.getElementById('sidebarToggle').addEventListener('click', function() {
      document.getElementById('sidebar').classList.toggle('active');
      document.querySelector('.main-content').classList.toggle('active');
    });
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(event) {
      const sidebar = document.getElementById('sidebar');
      const sidebarToggle = document.getElementById('sidebarToggle');
      
      if (window.innerWidth < 992 && 
          !sidebar.contains(event.target) && 
          !sidebarToggle.contains(event.target) && 
          sidebar.classList.contains('active')) {
        sidebar.classList.remove('active');
        document.querySelector('.main-content').classList.remove('active');
      }
    });
    
    // Enhanced delete confirmation
    function confirmDelete(csvNo, subject, className) {
      return confirm(`Are you sure you want to delete all records?\n\nCSV: ${csvNo}\nSubject: ${subject}\nClass: ${className}\n\nThis action cannot be undone!`);
    }
    
    // Export to CSV function
    function exportToCSV() {
      const table = document.querySelector('.results-table');
      let csv = [];
      
      // Get headers
      const headers = [];
      table.querySelectorAll('thead th').forEach(header => {
        if (header.textContent !== 'Actions') {
          headers.push(header.textContent);
        }
      });
      csv.push(headers.join(','));
      
      // Get data rows (skip action column)
      table.querySelectorAll('tbody tr').forEach(row => {
        const rowData = [];
        row.querySelectorAll('td').forEach((cell, index) => {
          // Skip the last column (Actions)
          if (index < row.cells.length - 1) {
            let text = cell.textContent.trim();
            // Remove badge text if present
            const badge = cell.querySelector('.badge-count');
            if (badge) {
              text = badge.textContent;
            }
            // Handle CSV formatting
            if (text.includes(',') || text.includes('"') || text.includes('\n')) {
              text = '"' + text.replace(/"/g, '""') + '"';
            }
            rowData.push(text);
          }
        });
        csv.push(rowData.join(','));
      });
      
      // Create and download CSV file
      const csvContent = csv.join('\n');
      const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
      const link = document.createElement('a');
      const url = URL.createObjectURL(blob);
      
      link.setAttribute('href', url);
      link.setAttribute('download', `uploaded_results_${new Date().toISOString().slice(0,10)}.csv`);
      link.style.visibility = 'hidden';
      
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
    }
    
    // Auto-refresh on delete success
    <?php if (isset($success_message) && isset($_GET['delete_csv'])): ?>
      // Remove success message after 5 seconds
      setTimeout(() => {
        const successAlert = document.querySelector('.alert-success');
        if (successAlert) {
          successAlert.style.opacity = '0';
          successAlert.style.transition = 'opacity 0.5s';
          setTimeout(() => {
            successAlert.remove();
          }, 500);
        }
      }, 5000);
    <?php endif; ?>
    
    // Print functionality
    function printTable() {
      window.print();
    }
  </script>
</body>
</html>