<?php
require_once('source/session.php');
require_once('source/error.php');
require_once('source/dbconfig.php');
require_once('source/function.php');
require_once('source/restrict_admin.php');
session_start();

// Check if the user is logged in
if (isset($_SESSION['SESS_ID_ADMIN'])) {
    $userId = $_SESSION['SESS_ID_ADMIN'];

    // Fetch user details based on the ID
    $query = "SELECT * FROM tbl_admin WHERE id = :id";
    $stmt = $DBH->prepare($query);
    $stmt->bindParam(':id', $userId, PDO::PARAM_INT);
    $stmt->execute();

    if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $adminName = $row['username'];
    }
}

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check if batch_id column exists, if not, add it
function checkAndAddBatchColumn() {
    global $DBH;
    try {
        // Check if batch_id column exists
        $stmt = $DBH->query("SHOW COLUMNS FROM tbl_pin LIKE 'batch_id'");
        $columnExists = $stmt->rowCount() > 0;
        
        if (!$columnExists) {
            // Add batch_id column if it doesn't exist
            $DBH->exec("ALTER TABLE tbl_pin ADD COLUMN batch_id VARCHAR(100) DEFAULT NULL");
        }
    } catch (PDOException $e) {
        // Log error but don't stop execution
        error_log("Error checking/adding batch_id column: " . $e->getMessage());
    }
}

// Call function to check column
checkAndAddBatchColumn();

// Function to generate random PINs
function generateRandomPIN($length = 10) {
    $pin = '';
    $characters = '0123456789';
    $char_length = strlen($characters);

    for ($i = 0; $i < $length; $i++) {
        $pin .= $characters[rand(0, $char_length - 1)];
    }

    return $pin;
}

// Handle PIN generation with batches
if (isset($_POST['generate'])) {
    $quantity = intval($_POST['quantity']);
    $batchName = trim($_POST['batch_name']);
    
    // Generate batch ID if not provided
    if (empty($batchName)) {
        $batchName = 'BATCH_' . date('Ymd_His');
    }
    
    if ($quantity <= 0) {
        $_SESSION['error'] = "Quantity must be greater than 0";
        header("Location: generate_pins.php");
        exit();
    }
    
    if ($quantity > 5000) {
        $_SESSION['error'] = "Maximum 5000 PINs can be generated at once";
        header("Location: generate_pins.php");
        exit();
    }
    
    try {
        $DBH->beginTransaction();
        
        // Get the maximum serial number for this batch
        $stmt = $DBH->prepare("SELECT MAX(Serial) FROM tbl_pin WHERE batch_id = ?");
        $stmt->execute([$batchName]);
        $maxSerial = $stmt->fetchColumn();
        $nextSerial = $maxSerial ? $maxSerial + 1 : 1;
        
        // Generate and insert the PINs into the database
        $stmt = $DBH->prepare("INSERT INTO tbl_pin (pin, Serial, batch_id, date_time) VALUES (?, ?, ?, NOW())");
        
        $generatedPins = [];
        for ($i = 0; $i < $quantity; $i++) {
            $pin = generateRandomPIN();
            $serial = $nextSerial + $i;
            $stmt->execute([$pin, $serial, $batchName]);
            $generatedPins[] = [
                'pin' => $pin,
                'serial' => $serial
            ];
        }
        
        $DBH->commit();
        $_SESSION['success'] = "Successfully generated $quantity PINs in batch '$batchName'";
        
        // Store generated pins in session for display
        $_SESSION['last_generated_batch'] = $batchName;
        $_SESSION['last_generated_pins'] = $generatedPins;
        
    } catch (PDOException $e) {
        $DBH->rollBack();
        $_SESSION['error'] = "Error generating PINs: " . $e->getMessage();
    }
    
    header("Location: generate_pins.php");
    exit();
}

// Handle batch deletion
if (isset($_POST['delete_batch'])) {
    $batchName = $_POST['batch_name'];
    
    try {
        $DBH->beginTransaction();
        $stmt = $DBH->prepare("DELETE FROM tbl_pin WHERE batch_id = ?");
        $stmt->execute([$batchName]);
        $DBH->commit();
        $_SESSION['success'] = "Batch '$batchName' deleted successfully";
    } catch (PDOException $e) {
        $DBH->rollBack();
        $_SESSION['error'] = "Error deleting batch: " . $e->getMessage();
    }
    
    header("Location: generate_pins.php");
    exit();
}

// Handle delete selected PINs
if (isset($_POST['delete_selected']) && isset($_POST['selected_pins'])) {
    $selectedPins = $_POST['selected_pins'];
    
    if (!empty($selectedPins)) {
        try {
            $DBH->beginTransaction();
            $placeholders = implode(',', array_fill(0, count($selectedPins), '?'));
            $stmt = $DBH->prepare("DELETE FROM tbl_pin WHERE id IN ($placeholders)");
            $stmt->execute($selectedPins);
            $DBH->commit();
            $_SESSION['success'] = count($selectedPins) . " PIN(s) deleted successfully";
        } catch (PDOException $e) {
            $DBH->rollBack();
            $_SESSION['error'] = "Error deleting PINs: " . $e->getMessage();
        }
    }
    
    header("Location: generate_pins.php");
    exit();
}

// Get all batches
try {
    $batches = sql($DBH, "SELECT batch_id, COUNT(*) as pin_count, MIN(date_time) as created_date 
                          FROM tbl_pin 
                          WHERE batch_id IS NOT NULL AND batch_id != ''
                          GROUP BY batch_id 
                          ORDER BY created_date DESC", [], "rows");
} catch (Exception $e) {
    $batches = [];
    error_log("Error getting batches: " . $e->getMessage());
}

// Get current batch (from GET parameter or last generated)
$currentBatch = isset($_GET['batch']) ? $_GET['batch'] : 
                (isset($_SESSION['last_generated_batch']) ? $_SESSION['last_generated_batch'] : null);

// Get pins based on selection
if ($currentBatch) {
    $rows1 = sql($DBH, "SELECT * FROM tbl_pin WHERE batch_id = ? ORDER BY Serial", [$currentBatch], "rows");
} else {
    $rows1 = sql($DBH, "SELECT * FROM tbl_pin ORDER BY id DESC LIMIT 100", [], "rows");
}

// Handle print request
if (isset($_GET['print']) && isset($_GET['batch'])) {
    $printBatch = $_GET['batch'];
    $printPins = sql($DBH, "SELECT * FROM tbl_pin WHERE batch_id = ? ORDER BY Serial", [$printBatch], "rows");
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
  <title>VERBUM DEI ACADEMY - Generate PINs</title>
  <link rel="icon" type="image/png" href="https://portal.vda.com.ng/pri_admin/pri_logo.jpg">
  
  <!-- Bootstrap 5 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  
  <!-- Font Awesome -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  
  <!-- Google Fonts - Nunito -->
  <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@400;600;700&display=swap" rel="stylesheet">
  
  <!-- Custom CSS -->
  <style>
    :root {
      --primary-color: #1976d2;
      --secondary-color: #64b5f6;
      --accent-color: #bbdefb;
      --dark-color: #0d47a1;
      --light-color: #e3f2fd;
      --text-dark: #212529;
      --text-light: #f8f9fa;
    }
    
    body {
      font-family: 'Nunito', sans-serif;
      background-color: #f5f7fa;
      padding-top: 56px;
      color: var(--text-dark);
      font-size: 0.9rem;
    }
    
    /* Sidebar Styles */
    .sidebar {
      background: linear-gradient(180deg, var(--dark-color) 0%, var(--primary-color) 100%);
      position: fixed;
      top: 56px;
      left: 0;
      width: 250px;
      height: calc(100vh - 56px);
      transition: all 0.3s;
      z-index: 1000;
      overflow-y: auto;
    }
    
    .sidebar .nav-link {
      color: rgba(255, 255, 255, 0.85);
      font-weight: 600;
      padding: 0.75rem 1rem;
      margin-bottom: 0.2rem;
      transition: all 0.2s;
    }
    
    .sidebar .nav-link:hover {
      color: #fff;
      background-color: rgba(255, 255, 255, 0.15);
    }
    
    .sidebar .nav-link.active {
      color: #fff;
      background-color: var(--secondary-color);
      box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    
    .sidebar .nav-link i {
      margin-right: 0.5rem;
      width: 20px;
      text-align: center;
    }
    
    /* Main Content Styles */
    .main-content {
      margin-left: 250px;
      width: calc(100% - 250px);
      transition: all 0.3s;
      min-height: calc(100vh - 56px);
      padding-bottom: 60px;
    }
    
    @media (max-width: 991.98px) {
      .sidebar {
        left: -250px;
      }
      .main-content {
        margin-left: 0;
        width: 100%;
      }
      body {
        padding-top: 56px;
      }
    }
    
    /* Navbar */
    .navbar {
      background-color: var(--primary-color);
      box-shadow: 0 0.15rem 1.75rem 0 rgba(33, 37, 41, 0.1);
      position: fixed;
      top: 0;
      width: 100%;
      z-index: 1030;
      color: white;
    }
    
    /* Welcome banner */
    .welcome-banner {
      background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
      color: white;
      border-radius: 0.5rem;
      padding: 0.75rem 1rem;
      margin: 1rem;
      box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    
    /* Form Styles */
    .form-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
    }
    
    /* Table Styles */
    .table-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
      overflow-x: auto;
    }
    
    /* PIN card styles */
    .pin-card {
      background: linear-gradient(135deg, #e3f2fd 0%, #ffffff 100%);
      border-left: 4px solid var(--primary-color);
      padding: 0.75rem;
      margin-bottom: 0.5rem;
      border-radius: 0 0.25rem 0.25rem 0;
      transition: transform 0.2s;
    }
    
    .pin-card:hover {
      transform: translateY(-2px);
      box-shadow: 0 4px 8px rgba(0,0,0,0.1);
    }
    
    .pin-code {
      font-family: 'Courier New', monospace;
      font-size: 1.2rem;
      font-weight: bold;
      letter-spacing: 2px;
      color: var(--dark-color);
      text-align: center;
      background-color: #f8f9fa;
      padding: 0.5rem;
      border-radius: 0.25rem;
      border: 1px solid #dee2e6;
      margin: 0.5rem 0;
    }
    
    /* Batch list styles */
    .batch-list {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
    }
    
    .batch-item {
      display: flex;
      justify-content: space-between;
      align-items: center;
      padding: 0.75rem;
      margin-bottom: 0.5rem;
      background-color: #f8f9fa;
      border-radius: 0.25rem;
      border-left: 4px solid var(--secondary-color);
      transition: all 0.2s;
    }
    
    .batch-item:hover {
      background-color: #e9ecef;
      transform: translateX(3px);
    }
    
    .batch-item.active {
      background-color: var(--light-color);
      border-left-color: var(--primary-color);
    }
    
    .batch-info {
      flex: 1;
    }
    
    .batch-actions {
      display: flex;
      gap: 0.5rem;
    }
    
    /* Footer styles */
    .footer {
      background-color: var(--dark-color);
      color: white;
      padding: 12px 0;
      position: fixed;
      bottom: 0;
      width: 100%;
      z-index: 100;
      font-size: 0.8rem;
    }
    
    .copyright-text {
      font-weight: 600;
    }
    
    .powered-by {
      display: flex;
      align-items: center;
      justify-content: flex-end;
      color: white;
    }
    
    .powered-by img {
      height: 18px;
      margin-left: 6px;
    }
    
    .powered-by-link {
      display: flex;
      align-items: center;
      text-decoration: none;
      color: inherit;
      transition: opacity 0.3s ease;
    }
    
    .powered-by-link:hover {
      opacity: 0.8;
      text-decoration: none;
    }
    
    /* Action buttons */
    .btn-action {
      padding: 0.5rem 1rem;
      border-radius: 0.25rem;
      font-weight: 600;
      transition: all 0.3s;
      font-size: 0.85rem;
    }
    
    .btn-primary {
      background-color: var(--primary-color);
      color: white;
      border: none;
    }
    
    .btn-primary:hover {
      background-color: var(--dark-color);
    }
    
    .btn-success {
      background-color: #28a745;
      color: white;
      border: none;
    }
    
    .btn-success:hover {
      background-color: #218838;
    }
    
    .btn-danger {
      background-color: #dc3545;
      color: white;
      border: none;
    }
    
    .btn-danger:hover {
      background-color: #c82333;
    }
    
    .btn-info {
      background-color: #17a2b8;
      color: white;
      border: none;
    }
    
    .btn-info:hover {
      background-color: #138496;
    }
    
    /* Alert styles */
    .alert {
      padding: 0.75rem 1rem;
      border-radius: 0.25rem;
      margin: 1rem;
    }
    
    .alert-success {
      background-color: #d4edda;
      border-color: #c3e6cb;
      color: #155724;
    }
    
    .alert-danger {
      background-color: #f8d7da;
      border-color: #f5c6cb;
      color: #721c24;
    }
    
    /* Compact form elements */
    .form-select, .form-label, .form-control {
      font-size: 0.85rem;
    }
    
    /* PIN grid */
    .pin-grid {
      display: grid;
      grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
      gap: 1rem;
      margin-top: 1rem;
    }
    
    /* Print styles */
    @media print {
      body * {
        visibility: hidden;
      }
      .print-section, .print-section * {
        visibility: visible;
      }
      .print-section {
        position: absolute;
        left: 0;
        top: 0;
        width: 100%;
        padding: 20px;
        background: white;
      }
      .no-print {
        display: none !important;
      }
      
      .pin-grid {
        grid-template-columns: repeat(3, 1fr);
        gap: 15px;
      }
      
      .pin-card {
        page-break-inside: avoid;
        border: 1px solid #000;
        box-shadow: none;
        border-radius: 5px;
        margin-bottom: 15px;
      }
    }
    
    /* Stats cards */
    .stats-container {
      display: flex;
      gap: 1rem;
      margin-bottom: 1rem;
      flex-wrap: wrap;
    }
    
    .stat-card {
      background: linear-gradient(135deg, #e3f2fd 0%, #f5f5f5 100%);
      padding: 0.75rem 1rem;
      border-radius: 0.25rem;
      text-align: center;
      flex: 1;
      min-width: 150px;
    }
    
    .stat-value {
      font-size: 1.5rem;
      font-weight: bold;
      color: var(--primary-color);
    }
    
    .stat-label {
      font-size: 0.75rem;
      color: #666;
    }
  </style>
</head>

<body>
  <!-- Top Navigation -->
  <nav class="navbar navbar-expand-lg navbar-light">
    <div class="container-fluid">
      <button id="sidebarToggle" class="btn btn-link d-lg-none me-2">
        <i class="fas fa-bars"></i>
      </button>
      <a class="navbar-brand d-none d-lg-block" href="#">
        VERBUM DEI ACADEMY
      </a>
      <div class="ms-auto d-flex align-items-center">
        <div class="admin-info me-3">
          <span class="admin-name"><?php echo htmlspecialchars($adminName ?? 'Admin'); ?></span>
          <span class="admin-access">(Administrator)</span>
          <i class="fas fa-user-circle admin-icon"></i>
        </div>
        <a href="logout.php" class="btn btn-sm btn-outline-light">
          <i class="fas fa-sign-out-alt"></i> Logout
        </a>
      </div>
    </div>
  </nav>

  <!-- Sidebar -->
  <nav id="sidebar" class="sidebar">
    <div class="position-sticky pt-3">
      <div class="text-center mb-4 d-none d-lg-block">
        <img src="https://portal.vda.com.ng/pri_admin/pri_logo.jpg" alt="School Logo" class="rounded-circle" width="80">
        <h5 class="mt-2 text-white">VERBUM DEI ACADEMY</h5>
        <p class="text-white-50 small">Admin Dashboard</p>
      </div>
      
      <ul class="nav flex-column">
        <li class="nav-item">
          <a class="nav-link" href="main.php">
            <i class="fas fa-fw fa-tachometer-alt"></i>
            Admin Dashboard
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link active" href="generate_pins.php">
            <i class="fas fa-fw fa-key"></i>
            Generate PINs
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="find_student_pin.php">
            <i class="fas fa-fw fa-search"></i>
            Find Student's PIN
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="pin_user.php">
            <i class="fas fa-fw fa-check-circle"></i>
            Confirm PIN Usage
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="edit_pin.php">
            <i class="fas fa-fw fa-edit"></i>
            Edit PIN
          </a>
        </li>
      </ul>
    </div>
  </nav>

  <!-- Main Content -->
  <main class="main-content">
    <!-- Welcome Banner -->
    <div class="welcome-banner">
      <div class="d-flex justify-content-between align-items-center">
        <div>
          <h4 class="mb-1">PIN Generation & Management</h4>
          <p class="mb-0 small">VERBUM DEI ACADEMY Admin Dashboard</p>
        </div>
        <div class="d-none d-sm-inline-block">
          <span class="badge bg-white text-primary p-1 small">
            <i class="fas fa-key me-1"></i> PIN System
          </span>
        </div>
      </div>
    </div>

    <?php if (isset($_SESSION['error'])): ?>
      <div class="alert alert-danger">
        <i class="fas fa-exclamation-circle me-2"></i> <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
      </div>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['success'])): ?>
      <div class="alert alert-success">
        <i class="fas fa-check-circle me-2"></i> <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
      </div>
    <?php endif; ?>

    <?php if (isset($_GET['print']) && isset($_GET['batch'])): ?>
      <!-- PRINT VIEW -->
      <div class="print-section">
        <div class="container-fluid">
          <div class="text-center mb-4">
            <h2>PIN BATCH REPORT</h2>
            <h3>Batch: <?php echo htmlspecialchars($printBatch); ?></h3>
            <p>Generated on: <?php echo date('F j, Y h:i A'); ?></p>
            <hr>
          </div>
          
          <div class="pin-grid">
            <?php 
            if (is_array($printPins) && count($printPins) > 0):
              foreach ($printPins as $pin): 
            ?>
              <div class="pin-card">
                <div class="d-flex justify-content-between align-items-center mb-2">
                  <span>Serial:</span>
                  <strong><?php echo isset($pin['Serial']) ? str_pad($pin['Serial'], 5, "0", STR_PAD_LEFT) : '-'; ?></strong>
                </div>
                <div class="d-flex justify-content-between align-items-center mb-2">
                  <span>PIN:</span>
                  <span class="pin-code"><?php echo isset($pin['pin']) ? $pin['pin'] : ''; ?></span>
                </div>
                <div class="text-center mt-3">
                  <small><?php echo htmlspecialchars($printBatch); ?></small>
                </div>
              </div>
            <?php 
              endforeach;
            else: ?>
              <div class="col-12">
                <p class="text-center">No PINs found in this batch</p>
              </div>
            <?php endif; ?>
          </div>
          
          <div class="text-center mt-4">
            <p><strong>Total PINs: <?php echo is_array($printPins) ? count($printPins) : 0; ?></strong></p>
          </div>
        </div>
      </div>
      
      <script>
        window.onload = function() {
          window.print();
          setTimeout(function() {
            window.close();
          }, 500);
        }
      </script>
      
    <?php else: ?>
      <!-- NORMAL VIEW -->
      
      <!-- Generate PINs Form -->
      <div class="form-container">
        <form method="post" action="">
          <div class="row align-items-end">
            <div class="col-md-5">
              <div class="form-group mb-2">
                <label for="batch_name" class="form-label">Batch Name (Optional):</label>
                <input type="text" class="form-control form-control-sm" id="batch_name" name="batch_name" 
                       placeholder="e.g., Term1-2023, Leave blank for auto-generated name">
              </div>
            </div>
            
            <div class="col-md-4">
              <div class="form-group mb-2">
                <label for="quantity" class="form-label">Number of PINs to Generate:</label>
                <input type="number" class="form-control form-control-sm" id="quantity" name="quantity" 
                       value="50" min="1" max="5000" required
                       placeholder="Enter number (1-5000)">
                <small class="form-text text-muted">Enter any number between 1 and 5000</small>
              </div>
            </div>
            
            <div class="col-md-3">
              <button type="submit" name="generate" class="btn btn-success btn-sm w-100">
                <i class="fas fa-plus me-1"></i> Generate PINs
              </button>
            </div>
          </div>
        </form>
      </div>

      <!-- Statistics - FIXED VERSION -->
      <div class="form-container">
        <div class="stats-container">
          <div class="stat-card">
            <div class="stat-value">
              <?php 
              $totalPinsResult = sql($DBH, "SELECT COUNT(*) FROM tbl_pin", [], "fetchColumn");
              $totalPins = is_numeric($totalPinsResult) ? intval($totalPinsResult) : 0;
              echo number_format($totalPins);
              ?>
            </div>
            <div class="stat-label">Total PINs Generated</div>
          </div>
          
          <div class="stat-card">
            <div class="stat-value">
              <?php 
              $usedPinsResult = sql($DBH, "SELECT COUNT(*) FROM tbl_pin WHERE pin_user IS NOT NULL AND pin_user != ''", [], "fetchColumn");
              $usedPins = is_numeric($usedPinsResult) ? intval($usedPinsResult) : 0;
              echo number_format($usedPins);
              ?>
            </div>
            <div class="stat-label">Used PINs</div>
          </div>
          
          <div class="stat-card">
            <div class="stat-value">
              <?php 
              $activeBatchesResult = sql($DBH, "SELECT COUNT(DISTINCT batch_id) FROM tbl_pin WHERE batch_id IS NOT NULL AND batch_id != ''", [], "fetchColumn");
              $activeBatches = is_numeric($activeBatchesResult) ? intval($activeBatchesResult) : 0;
              echo number_format($activeBatches);
              ?>
            </div>
            <div class="stat-label">Active Batches</div>
          </div>
        </div>
      </div>

      <!-- PIN Batches -->
      <div class="batch-list">
        <h5 class="mb-3">
          <i class="fas fa-list me-2"></i>PIN Batches
          <small class="text-muted">(<?php echo count($batches); ?> batches)</small>
        </h5>
        
        <?php if (is_array($batches) && count($batches) > 0): ?>
          <?php foreach ($batches as $batch): ?>
            <div class="batch-item <?php echo ($currentBatch == $batch['batch_id']) ? 'active' : ''; ?>">
              <div class="batch-info">
                <h6 class="mb-1">
                  <a href="generate_pins.php?batch=<?php echo urlencode($batch['batch_id']); ?>" class="text-decoration-none">
                    <?php echo htmlspecialchars($batch['batch_id']); ?>
                  </a>
                </h6>
                <p class="mb-0 small">
                  <i class="fas fa-key text-muted me-1"></i><?php echo $batch['pin_count']; ?> PINs • 
                  <i class="fas fa-calendar text-muted me-1"></i><?php echo date('M j, Y h:i A', strtotime($batch['created_date'])); ?>
                </p>
              </div>
              <div class="batch-actions">
                <a href="generate_pins.php?batch=<?php echo urlencode($batch['batch_id']); ?>" 
                   class="btn btn-info btn-sm">
                  <i class="fas fa-eye"></i>
                </a>
                <form method="post" action="" style="display: inline;">
                  <input type="hidden" name="batch_name" value="<?php echo htmlspecialchars($batch['batch_id']); ?>">
                  <button type="submit" name="delete_batch" class="btn btn-danger btn-sm" 
                          onclick="return confirm('Delete entire batch? This cannot be undone.');">
                    <i class="fas fa-trash"></i>
                  </button>
                </form>
                <a href="generate_pins.php?print=1&batch=<?php echo urlencode($batch['batch_id']); ?>" 
                   class="btn btn-success btn-sm" target="_blank">
                  <i class="fas fa-print"></i>
                </a>
              </div>
            </div>
          <?php endforeach; ?>
        <?php else: ?>
          <div class="text-center py-4">
            <i class="fas fa-key fa-3x mb-2 text-muted"></i>
            <h5 class="text-muted">No Batches Found</h5>
            <p class="mb-0">Generate your first batch of PINs using the form above.</p>
          </div>
        <?php endif; ?>
        
        <!-- View All PINs -->
        <div class="batch-item <?php echo (!$currentBatch) ? 'active' : ''; ?> mt-3">
          <div class="batch-info">
            <h6 class="mb-1">
              <a href="generate_pins.php" class="text-decoration-none">All PINs (Recent 100)</a>
            </h6>
            <p class="mb-0 small">View the most recent PINs generated</p>
          </div>
          <div class="batch-actions">
            <a href="generate_pins.php" class="btn btn-warning btn-sm">
              <i class="fas fa-list"></i>
            </a>
          </div>
        </div>
      </div>

      <!-- PIN Display Section -->
      <div class="table-container">
        <div class="d-flex justify-content-between align-items-center mb-3">
          <h5 class="mb-0">
            <i class="fas fa-key me-2"></i>
            <?php if ($currentBatch): ?>
              PIN Batch: <?php echo htmlspecialchars($currentBatch); ?>
              <small class="text-muted">(<?php echo count($rows1); ?> PINs)</small>
            <?php else: ?>
              Recent PINs
              <small class="text-muted">(Most recent 100)</small>
            <?php endif; ?>
          </h5>
          <?php if ($currentBatch): ?>
            <a href="generate_pins.php?print=1&batch=<?php echo urlencode($currentBatch); ?>" 
               class="btn btn-success btn-sm no-print" target="_blank">
              <i class="fas fa-print me-1"></i> Print Batch
            </a>
          <?php endif; ?>
        </div>
        
        <?php if (is_array($rows1) && count($rows1) > 0): ?>
          <div class="pin-grid">
            <?php foreach ($rows1 as $row1): ?>
              <div class="pin-card">
                <div class="d-flex justify-content-between align-items-center mb-2">
                  <span class="text-muted">Serial:</span>
                  <strong><?php echo isset($row1['Serial']) ? str_pad($row1['Serial'], 5, "0", STR_PAD_LEFT) : '-'; ?></strong>
                </div>
                <div class="d-flex justify-content-between align-items-center mb-2">
                  <span class="text-muted">PIN:</span>
                  <span class="pin-code"><?php echo $row1['pin']; ?></span>
                </div>
                <div class="text-center mt-3">
                  <small class="text-muted">
                    <?php if (isset($row1['batch_id'])): ?>
                      <i class="fas fa-folder me-1"></i><?php echo htmlspecialchars($row1['batch_id']); ?>
                    <?php else: ?>
                      <i class="fas fa-folder me-1"></i>No Batch
                    <?php endif; ?>
                  </small>
                </div>
              </div>
            <?php endforeach; ?>
          </div>
        <?php else: ?>
          <div class="text-center py-4">
            <i class="fas fa-key fa-3x mb-2 text-muted"></i>
            <h5 class="text-muted">No PINs Found</h5>
            <p class="mb-0">Generate some PINs to see them here.</p>
          </div>
        <?php endif; ?>
        
        <div class="mt-3 no-print">
          <div class="alert alert-info py-2">
            <h6 class="mb-1"><i class="fas fa-info-circle me-1"></i> PIN Information</h6>
            <p class="mb-0 small">
              1. Each PIN is 10 digits long and randomly generated.<br>
              2. PINs are grouped into batches for easier management.<br>
              3. You can print a batch to distribute to students.<br>
              4. Maximum of 5000 PINs can be generated at once.<br>
              5. PINs can be searched and verified in the PIN management section.
            </p>
          </div>
        </div>
      </div>
    <?php endif; ?>
  </main>

  <!-- Footer -->
  <footer class="footer no-print">
    <div class="container-fluid">
      <div class="row align-items-center">
        <div class="col-md-6">
          <span class="copyright-text">Copyright &copy; VERBUM DEI ACADEMY <?php echo date("Y"); ?></span>
        </div>
        <div class="col-md-6">
          <div class="powered-by">
            <a href="https://dataswift.com.ng" target="_blank" rel="noopener noreferrer" class="powered-by-link">
              <span>Powered by DataSwift</span>
              <img src="datalogo.png" alt="DataSwift Logo">
            </a>
          </div>
        </div>
      </div>
    </div>
  </footer>

  <!-- Bootstrap 5 JS Bundle with Popper -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
  
  <!-- Custom Scripts -->
  <script>
    // Toggle sidebar on mobile
    document.getElementById('sidebarToggle').addEventListener('click', function() {
      document.getElementById('sidebar').classList.toggle('active');
      document.querySelector('.main-content').classList.toggle('active');
    });
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(event) {
      const sidebar = document.getElementById('sidebar');
      const sidebarToggle = document.getElementById('sidebarToggle');
      
      if (window.innerWidth < 992 && 
          !sidebar.contains(event.target) && 
          !sidebarToggle.contains(event.target) && 
          sidebar.classList.contains('active')) {
        sidebar.classList.remove('active');
        document.querySelector('.main-content').classList.remove('active');
      }
    });
    
    // Auto-generate batch name if empty
    document.getElementById('batch_name').addEventListener('blur', function() {
      if (!this.value.trim()) {
        const now = new Date();
        const dateStr = now.toISOString().slice(0,10).replace(/-/g, '');
        const timeStr = now.toTimeString().slice(0,8).replace(/:/g, '');
        this.value = `BATCH_${dateStr}_${timeStr}`;
      }
    });
    
    // Validate quantity input
    document.getElementById('quantity').addEventListener('input', function() {
      const value = parseInt(this.value);
      if (value > 5000) {
        this.value = 5000;
        alert('Maximum 5000 PINs can be generated at once');
      } else if (value < 1) {
        this.value = 1;
      }
    });
  </script>
</body>
</html>