<?php
try {
    require_once('source/session.php');
    require_once('source/error.php');
    require_once('source/dbconfig.php');
    require_once('source/function.php');
    session_start();

    // Check if the user is logged in
    if (isset($_SESSION['SESS_ID_ADMIN'])) {
        $userId = $_SESSION['SESS_ID_ADMIN'];

        // Fetch user details based on the ID
        $query = "SELECT * FROM tbl_admin WHERE id = :id";
        $stmt = $DBH->prepare($query);
        $stmt->bindParam(':id', $userId, PDO::PARAM_INT);
        $stmt->execute();

        if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $adminName = $row['username'];
        }
    }

    // Fetch dynamic options for class, term, and year from the database
    $classes = sql($DBH, "SELECT DISTINCT class FROM attendance ORDER BY class", [], "rows");
    $terms = sql($DBH, "SELECT DISTINCT term FROM attendance ORDER BY term", [], "rows");
    $years = sql($DBH, "SELECT DISTINCT year FROM attendance ORDER BY year DESC", [], "rows");

    // Initialize variables
    $class = $_POST['class'] ?? '';
    $term = $_POST['term'] ?? '';
    $year = $_POST['year'] ?? '';
    $page_title = '';
    $total_results_count = 0;
    $comments_results = [];

    if ($_SERVER["REQUEST_METHOD"] == "POST") {
        // Extract search data from the form
        $class = strip_tags($_POST['class']);
        $term = strip_tags($_POST['term']);
        $year = strip_tags($_POST['year']);

        // Retrieve attendance records from the database based on the search criteria 
        $comments_results = sql($DBH, 
            "SELECT id, name, registration_no, class, year, term, 
                    Present, SchOpen, DateOfReport, NextTermBegins, 
                    TeachersRemarks, Absent 
             FROM attendance 
             WHERE class = ? AND term = ? AND year = ? 
             ORDER BY name",
            [$class, $term, $year], 
            "rows"
        );

        // Set page title
        $page_title = "Attendance & Comments Modification";

        // Count of total records
        $total_results_count = count($comments_results);

        // Handle form submissions for saving all fields
        if (isset($_POST['save_all'])) {
            $ids = $_POST['id'];
            $present_values = $_POST['present'];
            $schopen_values = $_POST['schopen'];
            $dateofreport_values = $_POST['dateofreport'];
            $nexttermbegins_values = $_POST['nexttermbegins'];
            $teachersremarks_values = $_POST['teachersremarks'];
            $absent_values = $_POST['absent'];

            $success_count = 0;
            $error_messages = [];
            
            foreach ($ids as $index => $id) {
                // Convert to integers and validate
                $present = isset($present_values[$index]) ? intval($present_values[$index]) : 0;
                $schopen = isset($schopen_values[$index]) ? intval($schopen_values[$index]) : 0;
                $absent = isset($absent_values[$index]) ? intval($absent_values[$index]) : 0;
                
                // Keep text fields as strings (varchar)
                $dateofreport = isset($dateofreport_values[$index]) ? strip_tags($dateofreport_values[$index]) : '';
                $nexttermbegins = isset($nexttermbegins_values[$index]) ? strip_tags($nexttermbegins_values[$index]) : '';
                $teachersremarks = isset($teachersremarks_values[$index]) ? strip_tags($teachersremarks_values[$index]) : '';
                
                // Validate numeric fields (allow up to 4 digits - 0-9999)
                if ($present >= 0 && $present <= 9999 && 
                    $schopen >= 0 && $schopen <= 9999 && 
                    $absent >= 0 && $absent <= 9999) {
                    
                    $update_query = "UPDATE attendance SET 
                                    Present = ?, 
                                    SchOpen = ?, 
                                    DateOfReport = ?, 
                                    NextTermBegins = ?, 
                                    TeachersRemarks = ?, 
                                    Absent = ? 
                                    WHERE id = ?";
                    
                    $stmt = $DBH->prepare($update_query);
                    if ($stmt->execute([$present, $schopen, $dateofreport, $nexttermbegins, $teachersremarks, $absent, $id])) {
                        $success_count++;
                    } else {
                        $error_messages[] = "Failed to update record ID: $id";
                    }
                } else {
                    $error_messages[] = "Invalid numeric values for record ID: $id (must be 0-9999)";
                }
            }

            if ($success_count > 0 && empty($error_messages)) {
                $success_message = "Successfully updated $success_count records";
                
                // Refresh the results
                $comments_results = sql($DBH, 
                    "SELECT id, name, registration_no, class, year, term, 
                            Present, SchOpen, DateOfReport, NextTermBegins, 
                            TeachersRemarks, Absent 
                     FROM attendance 
                     WHERE class = ? AND term = ? AND year = ? 
                     ORDER BY name",
                    [$class, $term, $year], 
                    "rows"
                );
            } elseif (!empty($error_messages)) {
                $error_message = implode("<br>", $error_messages);
            }
        }

        // Handle form submissions for deleting a record
        if (isset($_POST['delete']) && isset($_POST['delete_id'])) {
            $delete_id = strip_tags($_POST['delete_id']);
            
            // Get student name for confirmation message
            $student_info = sql($DBH, 
                "SELECT name FROM attendance WHERE id = ?", 
                [$delete_id], 
                "row"
            );
            
            $delete_query = "DELETE FROM attendance WHERE id = ?";
            $stmt = $DBH->prepare($delete_query);
            
            if ($stmt->execute([$delete_id])) {
                $success_message = "Successfully deleted record for: " . ($student_info['name'] ?? 'Unknown Student');
                
                // Refresh the results
                $comments_results = sql($DBH, 
                    "SELECT id, name, registration_no, class, year, term, 
                            Present, SchOpen, DateOfReport, NextTermBegins, 
                            TeachersRemarks, Absent 
                     FROM attendance 
                     WHERE class = ? AND term = ? AND year = ? 
                     ORDER BY name",
                    [$class, $term, $year], 
                    "rows"
                );
                $total_results_count = count($comments_results);
            } else {
                $error_message = "Failed to delete record";
            }
        }
    }
} catch (PDOException $e) {
    $error_message = "Database Error: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1, shrink-to-fit=no">
  <title>VERBUM DEI ACADEMY - Attendance & Comments Management</title>
  <link rel="icon" type="image/png" href="https://portal.vda.com.ng/pri_admin/pri_logo.jpg">
  
  <!-- Bootstrap 5 CSS -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
  
  <!-- Font Awesome -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  
  <!-- Google Fonts - Nunito -->
  <link href="https://fonts.googleapis.com/css2?family=Nunito:wght@400;600;700&display=swap" rel="stylesheet">
  
  <!-- Custom CSS -->
  <style>
    :root {
      --primary-color: #1976d2;
      --secondary-color: #64b5f6;
      --accent-color: #bbdefb;
      --dark-color: #0d47a1;
      --light-color: #e3f2fd;
      --text-dark: #212529;
      --text-light: #f8f9fa;
    }
    
    body {
      font-family: 'Nunito', sans-serif;
      background-color: #f5f7fa;
      padding-top: 56px;
      color: var(--text-dark);
      font-size: 0.9rem;
    }
    
    /* Sidebar Styles */
    .sidebar {
      background: linear-gradient(180deg, var(--dark-color) 0%, var(--primary-color) 100%);
      position: fixed;
      top: 56px;
      left: 0;
      width: 250px;
      height: calc(100vh - 56px);
      transition: all 0.3s;
      z-index: 1000;
      overflow-y: auto;
    }
    
    .sidebar .nav-link {
      color: rgba(255, 255, 255, 0.85);
      font-weight: 600;
      padding: 0.75rem 1rem;
      margin-bottom: 0.2rem;
      transition: all 0.2s;
    }
    
    .sidebar .nav-link:hover {
      color: #fff;
      background-color: rgba(255, 255, 255, 0.15);
    }
    
    .sidebar .nav-link.active {
      color: #fff;
      background-color: var(--secondary-color);
      box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    
    .sidebar .nav-link i {
      margin-right: 0.5rem;
      width: 20px;
      text-align: center;
    }
    
    /* Main Content Styles */
    .main-content {
      margin-left: 250px;
      width: calc(100% - 250px);
      transition: all 0.3s;
      min-height: calc(100vh - 56px);
      padding-bottom: 60px;
    }
    
    @media (max-width: 991.98px) {
      .sidebar {
        left: -250px;
      }
      .main-content {
        margin-left: 0;
        width: 100%;
      }
      body {
        padding-top: 56px;
      }
    }
    
    /* Navbar */
    .navbar {
      background-color: var(--primary-color);
      box-shadow: 0 0.15rem 1.75rem 0 rgba(33, 37, 41, 0.1);
      position: fixed;
      top: 0;
      width: 100%;
      z-index: 1030;
      color: white;
    }
    
    /* Welcome banner */
    .welcome-banner {
      background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
      color: white;
      border-radius: 0.5rem;
      padding: 0.75rem 1rem;
      margin: 1rem;
      box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    
    /* Form Styles */
    .form-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
    }
    
    /* Table Styles */
    .table-container {
      background-color: white;
      border-radius: 0.5rem;
      padding: 1rem;
      margin: 1rem;
      box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.1);
      overflow-x: auto;
    }
    
    .attendance-table {
      width: 100%;
      border-collapse: collapse;
      font-size: 0.85rem;
    }
    
    .attendance-table th,
    .attendance-table td {
      padding: 0.5rem;
      border: 1px solid #dee2e6;
      text-align: center;
      vertical-align: middle;
    }
    
    .attendance-table th {
      background-color: var(--primary-color);
      color: white;
      font-weight: 600;
      position: sticky;
      top: 0;
      z-index: 10;
    }
    
    .attendance-table input,
    .attendance-table textarea {
      width: 100%;
      padding: 0.25rem 0.5rem;
      border: 1px solid #ced4da;
      border-radius: 0.25rem;
      font-size: 0.85rem;
    }
    
    .attendance-table input:focus,
    .attendance-table textarea:focus {
      border-color: var(--secondary-color);
      box-shadow: 0 0 0 0.2rem rgba(100, 181, 246, 0.25);
      outline: none;
    }
    
    /* Specific input widths */
    .attendance-table .attendance-number {
      width: 70px;  /* Fixed width for number inputs */
    }
    
    .attendance-table .date-text {
      width: 120px;  /* Fixed width for date text inputs */
    }
    
    .attendance-table .student-name {
      text-align: left;
      font-weight: 600;
      background-color: var(--light-color);
      position: sticky;
      left: 0;
      z-index: 5;
      min-width: 120px;
    }
    
    .attendance-table .attendance-col {
      background-color: #e8f5e9;
      min-width: 80px;  /* Increased for 4-digit numbers */
    }
    
    .attendance-table .date-col {
      background-color: #fff3e0;
      min-width: 120px;  /* Increased for text dates */
    }
    
    .attendance-table .comment-col {
      background-color: #f3e5f5;
      min-width: 200px;
    }
    
    .attendance-table tr:nth-child(even) {
      background-color: #f9f9f9;
    }
    
    .attendance-table tr:hover {
      background-color: #f1f1f1;
    }
    
    /* Footer styles */
    .footer {
      background-color: var(--dark-color);
      color: white;
      padding: 12px 0;
      position: fixed;
      bottom: 0;
      width: 100%;
      z-index: 100;
      font-size: 0.8rem;
    }
    
    .copyright-text {
      font-weight: 600;
    }
    
    .powered-by {
      display: flex;
      align-items: center;
      justify-content: flex-end;
      color: white;
    }
    
    .powered-by img {
      height: 18px;
      margin-left: 6px;
    }
    
    .powered-by-link {
      display: flex;
      align-items: center;
      text-decoration: none;
      color: inherit;
      transition: opacity 0.3s ease;
    }
    
    .powered-by-link:hover {
      opacity: 0.8;
      text-decoration: none;
    }
    
    /* Action buttons */
    .btn-action {
      padding: 0.5rem 1rem;
      border-radius: 0.25rem;
      font-weight: 600;
      transition: all 0.3s;
      font-size: 0.85rem;
    }
    
    .btn-primary {
      background-color: var(--primary-color);
      color: white;
      border: none;
    }
    
    .btn-primary:hover {
      background-color: var(--dark-color);
    }
    
    .btn-success {
      background-color: #28a745;
      color: white;
      border: none;
    }
    
    .btn-success:hover {
      background-color: #218838;
    }
    
    .btn-danger {
      background-color: #dc3545;
      color: white;
      border: none;
    }
    
    .btn-danger:hover {
      background-color: #c82333;
    }
    
    /* Alert styles */
    .alert {
      padding: 0.75rem 1rem;
      border-radius: 0.25rem;
      margin: 1rem;
    }
    
    .alert-success {
      background-color: #d4edda;
      border-color: #c3e6cb;
      color: #155724;
    }
    
    .alert-danger {
      background-color: #f8d7da;
      border-color: #f5c6cb;
      color: #721c24;
    }
    
    /* Info cards */
    .info-card {
      background: linear-gradient(135deg, var(--light-color) 0%, #ffffff 100%);
      border-left: 4px solid var(--primary-color);
      padding: 0.75rem;
      margin-bottom: 0.5rem;
      border-radius: 0 0.25rem 0.25rem 0;
    }
    
    /* Compact form elements */
    .form-select, .form-label {
      font-size: 0.85rem;
    }
    
    /* Attendance stats */
    .attendance-stats {
      display: flex;
      gap: 1rem;
      margin-bottom: 1rem;
    }
    
    .stat-card {
      background: linear-gradient(135deg, #e3f2fd 0%, #f5f5f5 100%);
      padding: 0.5rem 1rem;
      border-radius: 0.25rem;
      text-align: center;
      flex: 1;
    }
    
    .stat-value {
      font-size: 1.5rem;
      font-weight: bold;
      color: var(--primary-color);
    }
    
    .stat-label {
      font-size: 0.75rem;
      color: #666;
    }
  </style>
</head>

<body>
  <!-- Top Navigation -->
  <nav class="navbar navbar-expand-lg navbar-light">
    <div class="container-fluid">
      <button id="sidebarToggle" class="btn btn-link d-lg-none me-2">
        <i class="fas fa-bars"></i>
      </button>
      <a class="navbar-brand d-none d-lg-block" href="#">
        VERBUM DEI ACADEMY
      </a>
      <div class="ms-auto d-flex align-items-center">
        <div class="admin-info me-3">
          <span class="admin-name"><?php echo htmlspecialchars($adminName ?? 'Admin'); ?></span>
          <span class="admin-access">(Administrator)</span>
          <i class="fas fa-user-circle admin-icon"></i>
        </div>
        <a href="logout.php" class="btn btn-sm btn-outline-light">
          <i class="fas fa-sign-out-alt"></i> Logout
        </a>
      </div>
    </div>
  </nav>

  <!-- Sidebar -->
  <nav id="sidebar" class="sidebar">
    <div class="position-sticky pt-3">
      <div class="text-center mb-4 d-none d-lg-block">
        <img src="https://portal.vda.com.ng/pri_admin/pri_logo.jpg" alt="School Logo" class="rounded-circle" width="80">
        <h5 class="mt-2 text-white">VERBUM DEI ACADEMY</h5>
        <p class="text-white-50 small">Admin Dashboard</p>
      </div>
      
      <ul class="nav flex-column">
        <li class="nav-item">
          <a class="nav-link" href="main.php">
            <i class="fas fa-fw fa-tachometer-alt"></i>
            Admin Dashboard
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="edit_scores_subject.php">
            <i class="fas fa-fw fa-edit"></i>
            Edit Subject Scores
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="download_broadsheet.php">
            <i class="fas fa-fw fa-file-pdf"></i>
            Result Broadsheet
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link" href="list_uploaded_results.php">
            <i class="fas fa-fw fa-list"></i>
            Uploaded Results
          </a>
        </li>
        <li class="nav-item">
          <a class="nav-link active" href="edit_comments_attendance.php">
            <i class="fas fa-fw fa-comments"></i>
            Attendance & Comments
          </a>
        </li>
      </ul>
    </div>
  </nav>

  <!-- Main Content -->
  <main class="main-content">
    <!-- Welcome Banner -->
    <div class="welcome-banner">
      <div class="d-flex justify-content-between align-items-center">
        <div>
          <h4 class="mb-1">Attendance & Comments Management</h4>
          <p class="mb-0 small">VERBUM DEI ACADEMY Admin Dashboard</p>
        </div>
        <div class="d-none d-sm-inline-block">
          <span class="badge bg-white text-primary p-1 small">
            <i class="fas fa-calendar-alt me-1"></i> Attendance System
          </span>
        </div>
      </div>
    </div>

    <!-- Search Form -->
    <div class="form-container">
      <form method="POST" action="">
        <div class="row align-items-end">
          <div class="col-md-4">
            <div class="form-group mb-2">
              <label for="class" class="form-label">Class:</label>
              <select id="class" name="class" class="form-select form-select-sm" required>
                <option value="">Select Class</option>
                <?php foreach ($classes as $classOption): ?>
                  <option value="<?php echo htmlspecialchars($classOption['class']); ?>" 
                    <?php echo ($classOption['class'] === $class) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($classOption['class']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="term" class="form-label">Term:</label>
              <select id="term" name="term" class="form-select form-select-sm" required>
                <option value="">Select Term</option>
                <?php foreach ($terms as $termOption): ?>
                  <option value="<?php echo htmlspecialchars($termOption['term']); ?>" 
                    <?php echo ($termOption['term'] === $term) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($termOption['term']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-3">
            <div class="form-group mb-2">
              <label for="year" class="form-label">Academic Year:</label>
              <select id="year" name="year" class="form-select form-select-sm" required>
                <option value="">Select Year</option>
                <?php foreach ($years as $yearOption): ?>
                  <option value="<?php echo htmlspecialchars($yearOption['year']); ?>" 
                    <?php echo ($yearOption['year'] === $year) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars($yearOption['year']); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
          
          <div class="col-md-2">
            <button type="submit" class="btn btn-primary btn-sm w-100">
              <i class="fas fa-search me-1"></i> Search Records
            </button>
          </div>
        </div>
      </form>
    </div>

    <?php if (isset($error_message)): ?>
      <div class="alert alert-danger">
        <i class="fas fa-exclamation-circle me-2"></i> <?php echo $error_message; ?>
      </div>
    <?php endif; ?>

    <?php if (isset($success_message)): ?>
      <div class="alert alert-success">
        <i class="fas fa-check-circle me-2"></i> <?php echo $success_message; ?>
      </div>
    <?php endif; ?>

    <?php if (!empty($page_title) && !empty($class) && !empty($term) && !empty($year)): ?>
      <div class="form-container">
        <div class="row">
          <div class="col-md-8">
            <div class="info-card">
              <h6 class="mb-1"><i class="fas fa-edit me-2"></i>Modifying Attendance & Comments For:</h6>
              <p class="mb-0">Class: <?php echo htmlspecialchars($class); ?>, Term: <?php echo htmlspecialchars($term); ?>, Year: <?php echo htmlspecialchars($year); ?></p>
            </div>
          </div>
          <div class="col-md-4">
            <div class="info-card">
              <h6 class="mb-1"><i class="fas fa-users me-2"></i>Total Records: <?php echo $total_results_count; ?></h6>
              <p class="mb-0 small">Number of student records found</p>
            </div>
          </div>
        </div>
        
        <?php if (!empty($comments_results)): 
          // Calculate attendance statistics
          $total_present = 0;
          $total_absent = 0;
          $total_schopen = 0;
          
          foreach ($comments_results as $record) {
            $total_present += intval($record['Present']);
            $total_absent += intval($record['Absent']);
            $total_schopen += intval($record['SchOpen']);
          }
        ?>
          <div class="attendance-stats">
            <div class="stat-card">
              <div class="stat-value"><?php echo $total_present; ?></div>
              <div class="stat-label">Total Present</div>
            </div>
            <div class="stat-card">
              <div class="stat-value"><?php echo $total_absent; ?></div>
              <div class="stat-label">Total Absent</div>
            </div>
            <div class="stat-card">
              <div class="stat-value"><?php echo $total_schopen; ?></div>
              <div class="stat-label">School Opening Days</div>
            </div>
          </div>
        <?php endif; ?>
      </div>
    <?php endif; ?>

    <?php if (!empty($comments_results)): ?>
      <div class="table-container">
        <form method="POST">
          <input type="hidden" name="class" value="<?php echo htmlspecialchars($class); ?>">
          <input type="hidden" name="term" value="<?php echo htmlspecialchars($term); ?>">
          <input type="hidden" name="year" value="<?php echo htmlspecialchars($year); ?>">
          
          <div class="d-flex justify-content-between align-items-center mb-3">
            <h5 class="mb-0">
              Student Attendance & Comments
              <small class="text-muted">(<?php echo $total_results_count; ?> records)</small>
            </h5>
            <div>
              <button type="submit" name="save_all" class="btn btn-success btn-sm">
                <i class="fas fa-save me-1"></i> Save All Changes
              </button>
              <a href="main.php" class="btn btn-primary btn-sm ms-2">
                <i class="fas fa-arrow-left me-1"></i> Back to Dashboard
              </a>
            </div>
          </div>
          
          <div class="table-responsive">
            <table class="attendance-table">
              <thead>
                <tr>
                  <th width="5%">S/N</th>
                  <th class="student-name">Student Name</th>
                  <th>Reg No</th>
                  <th class="attendance-col">Present</th>
                  <th class="attendance-col">Absent</th>
                  <th class="attendance-col">School Open Days</th>
                  <th class="date-col">Date of Report</th>
                  <th class="date-col">Next Term Begins</th>
                  <th class="comment-col">Teacher's Remarks</th>
                  <th width="10%">Actions</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($comments_results as $index => $record): ?>
                  <tr>
                    <td><?php echo $index + 1; ?></td>
                    <td class="student-name"><?php echo htmlspecialchars($record['name']); ?></td>
                    <td><?php echo htmlspecialchars($record['registration_no']); ?></td>
                    <td class="attendance-col">
                      <input type="number" name="present[]" 
                             class="attendance-number"
                             value="<?php echo htmlspecialchars($record['Present']); ?>" 
                             min="0" max="9999" step="1" required
                             placeholder="0-9999">
                    </td>
                    <td class="attendance-col">
                      <input type="number" name="absent[]" 
                             class="attendance-number"
                             value="<?php echo htmlspecialchars($record['Absent']); ?>" 
                             min="0" max="9999" step="1" required
                             placeholder="0-9999">
                    </td>
                    <td class="attendance-col">
                      <input type="number" name="schopen[]" 
                             class="attendance-number"
                             value="<?php echo htmlspecialchars($record['SchOpen']); ?>" 
                             min="0" max="9999" step="1" required
                             placeholder="0-9999">
                    </td>
                    <td class="date-col">
                      <input type="text" name="dateofreport[]" 
                             class="date-text"
                             value="<?php echo htmlspecialchars($record['DateOfReport']); ?>"
                             placeholder="e.g., 12th March, 2024">
                    </td>
                    <td class="date-col">
                      <input type="text" name="nexttermbegins[]" 
                             class="date-text"
                             value="<?php echo htmlspecialchars($record['NextTermBegins']); ?>"
                             placeholder="e.g., 15th April, 2024">
                    </td>
                    <td class="comment-col">
                      <textarea name="teachersremarks[]" rows="2" 
                                placeholder="Enter teacher's remarks..."><?php echo htmlspecialchars($record['TeachersRemarks']); ?></textarea>
                    </td>
                    <td>
                      <input type="hidden" name="id[]" value="<?php echo $record['id']; ?>">
                      <div class="btn-group btn-group-sm" role="group">
                        <button type="button" class="btn btn-danger" 
                                onclick="confirmDelete(<?php echo $record['id']; ?>, '<?php echo addslashes($record['name']); ?>')"
                                title="Delete Record">
                          <i class="fas fa-trash"></i>
                        </button>
                      </div>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>
          
          <div class="mt-3">
            <button type="submit" name="save_all" class="btn btn-success btn-action">
              <i class="fas fa-save me-1"></i> Save All Changes
            </button>
            <button type="button" class="btn btn-secondary btn-action ms-2" onclick="resetForm()">
              <i class="fas fa-undo me-1"></i> Reset Changes
            </button>
          </div>
          
          <!-- Hidden delete form -->
          <input type="hidden" name="delete_id" id="delete_id" value="">
        </form>
        
        <div class="mt-3">
          <div class="alert alert-info py-2">
            <h6 class="mb-1"><i class="fas fa-info-circle me-1"></i> Instructions</h6>
            <p class="mb-0 small">
              1. Present, Absent, and School Open Days accept numbers 0-9999.<br>
              2. Date fields are text inputs - enter in any format (e.g., "12th March, 2024").<br>
              3. Teacher's Remarks can be any text.<br>
              4. Click "Save All Changes" to update all records at once.<br>
              5. Use the delete button to remove individual records.
            </p>
          </div>
          
          <div class="alert alert-warning py-2">
            <h6 class="mb-1"><i class="fas fa-lightbulb me-1"></i> Date Format Suggestions</h6>
            <p class="mb-0 small">
              For Date of Report and Next Term Begins, you can use formats like:<br>
              • "12th March, 2024"<br>
              • "15 April 2024"<br>
              • "March 12, 2024"<br>
              • "12/03/2024"<br>
              Use whatever format your school prefers!
            </p>
          </div>
        </div>
      </div>
    <?php elseif ($_SERVER["REQUEST_METHOD"] == "POST" && empty($comments_results)): ?>
      <div class="form-container text-center py-4">
        <i class="fas fa-search fa-3x mb-2 text-muted"></i>
        <h5 class="text-muted">No Records Found</h5>
        <p class="mb-0">There are no attendance records for the selected criteria.</p>
      </div>
    <?php else: ?>
      <div class="form-container text-center py-4">
        <i class="fas fa-comments fa-3x mb-2 text-muted"></i>
        <h5 class="text-muted">Attendance & Comments Management</h5>
        <p class="mb-0">Select a class, term, and academic year to edit attendance and comments.</p>
      </div>
    <?php endif; ?>
  </main>

  <!-- Footer -->
  <footer class="footer">
    <div class="container-fluid">
      <div class="row align-items-center">
        <div class="col-md-6">
          <span class="copyright-text">Copyright &copy; VERBUM DEI ACADEMY <?php echo date("Y"); ?></span>
        </div>
        <div class="col-md-6">
          <div class="powered-by">
            <a href="https://dataswift.com.ng" target="_blank" rel="noopener noreferrer" class="powered-by-link">
              <span>Powered by DataSwift</span>
              <img src="datalogo.png" alt="DataSwift Logo">
            </a>
          </div>
        </div>
      </div>
    </div>
  </footer>

  <!-- Bootstrap 5 JS Bundle with Popper -->
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
  
  <!-- Custom Scripts -->
  <script>
    // Toggle sidebar on mobile
    document.getElementById('sidebarToggle').addEventListener('click', function() {
      document.getElementById('sidebar').classList.toggle('active');
      document.querySelector('.main-content').classList.toggle('active');
    });
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(event) {
      const sidebar = document.getElementById('sidebar');
      const sidebarToggle = document.getElementById('sidebarToggle');
      
      if (window.innerWidth < 992 && 
          !sidebar.contains(event.target) && 
          !sidebarToggle.contains(event.target) && 
          sidebar.classList.contains('active')) {
        sidebar.classList.remove('active');
        document.querySelector('.main-content').classList.remove('active');
      }
    });
    
    // Delete confirmation function
    function confirmDelete(id, name) {
      if (confirm(`Are you sure you want to delete the record for:\n\n${name}?\n\nThis action cannot be undone!`)) {
        document.getElementById('delete_id').value = id;
        // Create a form to submit the delete request
        const form = document.createElement('form');
        form.method = 'POST';
        form.style.display = 'none';
        
        // Add hidden inputs
        const classInput = document.createElement('input');
        classInput.type = 'hidden';
        classInput.name = 'class';
        classInput.value = '<?php echo htmlspecialchars($class); ?>';
        form.appendChild(classInput);
        
        const termInput = document.createElement('input');
        termInput.type = 'hidden';
        termInput.name = 'term';
        termInput.value = '<?php echo htmlspecialchars($term); ?>';
        form.appendChild(termInput);
        
        const yearInput = document.createElement('input');
        yearInput.type = 'hidden';
        yearInput.name = 'year';
        yearInput.value = '<?php echo htmlspecialchars($year); ?>';
        form.appendChild(yearInput);
        
        const deleteIdInput = document.createElement('input');
        deleteIdInput.type = 'hidden';
        deleteIdInput.name = 'delete_id';
        deleteIdInput.value = id;
        form.appendChild(deleteIdInput);
        
        const deleteInput = document.createElement('input');
        deleteInput.type = 'hidden';
        deleteInput.name = 'delete';
        deleteInput.value = '1';
        form.appendChild(deleteInput);
        
        document.body.appendChild(form);
        form.submit();
      }
    }
    
    // Reset form function
    function resetForm() {
      if (confirm('Are you sure you want to reset all changes?')) {
        location.reload();
      }
    }
    
    // Validate numeric inputs (0-9999)
    document.querySelectorAll('.attendance-number').forEach(input => {
      input.addEventListener('blur', function() {
        const value = parseInt(this.value);
        if (isNaN(value) || value < 0 || value > 9999) {
          alert('Please enter a number between 0 and 9999');
          this.value = '';
          this.focus();
        }
      });
      
      input.addEventListener('input', function() {
        const row = this.closest('tr');
        const present = parseInt(row.querySelector('input[name="present[]"]').value) || 0;
        const absent = parseInt(row.querySelector('input[name="absent[]"]').value) || 0;
        const schopen = parseInt(row.querySelector('input[name="schopen[]"]').value) || 0;
        
        // Validate that present + absent doesn't exceed school open days (if both are set)
        if (schopen > 0 && (present + absent) > schopen) {
          alert('Present + Absent cannot exceed School Open Days');
          this.value = '';
          this.focus();
        }
      });
    });
    
    // Add placeholder suggestions for date fields on focus
    document.querySelectorAll('.date-text').forEach(input => {
      input.addEventListener('focus', function() {
        if (!this.value) {
          const today = new Date();
          const options = { day: 'numeric', month: 'long', year: 'numeric' };
          const formattedDate = today.toLocaleDateString('en-GB', options);
          this.placeholder = 'e.g., ' + formattedDate.replace(/(\d+)/, (match) => {
            const day = parseInt(match);
            const suffix = ['th', 'st', 'nd', 'rd'][(day % 10 > 3 || (day % 100 > 10 && day % 100 < 14)) ? 0 : day % 10] || 'th';
            return day + suffix;
          });
        }
      });
    });
    
    // Focus first input field if records exist
    <?php if (!empty($comments_results)): ?>
      document.addEventListener('DOMContentLoaded', function() {
        const firstInput = document.querySelector('input[type="number"]');
        if (firstInput) {
          firstInput.focus();
        }
      });
    <?php endif; ?>
  </script>
</body>
</html>