<?php
require_once('source/session.php');
require_once('source/error.php');
require_once('source/dbconfig.php');
require_once('source/function.php');
require_once('source/restrict_user.php');

// Initialize message variable
$msg = '';
if (isset($_GET['msg'])) {
    $messages = [
        'refered' => "<div class='alert alert-success'>Referred successfully</div>",
        'failed' => "<div class='alert alert-warning'>Something went wrong, please try again</div>",
    ];
    $msg = $messages[$_GET['msg']] ?? '';
}

if (isset($_SESSION['msg']) && $_SESSION['msg'] == "update") {
    $msg = "<div class='alert alert-success text-center'>Order Update Successful. Customer will receive email. <a href='new_order.php' class='btn btn-sm btn-outline-success'>Refresh</a></div>";
    unset($_SESSION['msg']);
}

// Validate and sanitize input parameters
$registration_no = $_GET['myreg'] ?? '';
$myclass = trim($_GET['myclass'] ?? '');
$term = trim($_GET['term'] ?? '');
$examyear = trim($_GET['examyear'] ?? '');

if (empty($registration_no) || empty($myclass) || empty($term) || empty($examyear)) {
    die("<div class='alert alert-danger'>Required parameters are missing.</div>");
}

// Check if class is Pre-Nursery or Reception
function isPreNurseryClass($className) {
    $className = strtoupper(trim($className));
    return (strpos($className, 'PRE-NUR') === 0 || strpos($className, 'RECEPTION') === 0);
}

if (!isPreNurseryClass($myclass)) {
    die("<div class='alert alert-danger'>This page is for Pre-Nursery and Reception classes only.</div>");
}

// Fetch student details
$student = sql($DBH, "SELECT * FROM tbl_user WHERE registration_no = ?", [$registration_no], "rows");
if (empty($student)) {
    die("<div class='alert alert-danger'>Student not found.</div>");
}
$student = $student[0];

$user_name = $student['fullname'];
$user_gender = $student['gender'];
$user_age = $student['age'];
$user_profile_pic = $student['profile_pic'] ?: 'default';
$profile_pic = $user_profile_pic == 'default' 
    ? "<img class='img-fluid rounded-circle profile-pic' width='100' src='../pri_admin/profile/profile.jpg' alt='Profile'>" 
    : "<img class='img-fluid rounded-circle profile-pic' width='100' src='../pri_admin/profile_pic/$user_profile_pic' alt='Profile'>";

// Calculate age
$age = '';
if (!empty($user_age)) {
    if (strpos($user_age, '/') !== false) {
        $birthDate = explode('/', $user_age);
        $month = (int)$birthDate[0];
        $day = (int)$birthDate[1];
        $year = (int)$birthDate[2];
    } elseif (strpos($user_age, '-') !== false) {
        $birthDate = explode('-', $user_age);
        $year = (int)$birthDate[0];
        $month = (int)$birthDate[1];
        $day = (int)$birthDate[2];
    } else {
        $month = 1;
        $day = 1;
        $year = 2000;
    }

    $currentYear = date('Y');
    $currentMonth = date('m');
    $currentDay = date('d');

    $age = $currentYear - $year;
    
    if ($currentMonth < $month || ($currentMonth == $month && $currentDay < $day)) {
        $age--;
    }
} else {
    $age = 'N/A';
}

// Fetch Pre-Nursery comments for this student
$comments = sql($DBH, "SELECT * FROM pre_nur_comments WHERE registration_no = ? AND class = ? AND term = ? AND year = ? ORDER BY subject", 
    [$registration_no, $myclass, $term, $examyear], "rows");

// Fetch class teacher
$class_teacher = sql($DBH, "SELECT fullname FROM tbl_user WHERE class = ? AND teacher = 'true' LIMIT 1", [$myclass], "rows");
$teacher_name = !empty($class_teacher) ? $class_teacher[0]['fullname'] : 'Class Teacher';

// Fetch attendance
$attendance = sql($DBH, "SELECT * FROM attendance WHERE registration_no = ? AND class = ? AND term = ? AND year = ?", 
    [$registration_no, $myclass, $term, $examyear], "rows")[0] ?? [];
$SchOpen = $attendance['SchOpen'] ?? 0;
$Present = $attendance['Present'] ?? 0;
$Absent = $SchOpen - $Present;
$DateOfReport = $attendance['DateOfReport'] ?? '';
$NextTermBegins = $attendance['NextTermBegins'] ?? '';
$TeachersRemarks = $attendance['TeachersRemarks'] ?? '';
$PrincipalsComment = $attendance['PrincipalsComment'] ?? '';

// Count number of Pre-Nursery students in class
try {
    $stmt = $DBH->prepare("SELECT COUNT(DISTINCT registration_no) AS student_count FROM pre_nur_comments WHERE TRIM(class) = ? AND TRIM(term) = ? AND TRIM(year) = ?");
    $stmt->execute([$myclass, $term, $examyear]);
    $count_student = (int)$stmt->fetchColumn();
    if ($count_student === 0) {
        $count_student = 1; // At least this student
    }
} catch (PDOException $e) {
    $count_student = 1;
}

// Get all comments for this class to show distribution
$all_class_comments = sql($DBH, "SELECT COUNT(*) as count, teacher_name FROM pre_nur_comments WHERE class = ? AND term = ? AND year = ? GROUP BY teacher_name ORDER BY teacher_name", 
    [$myclass, $term, $examyear], "rows");

// Get overall teacher comments for Pre-Nursery
$overall_teacher_comments = sql($DBH, "SELECT * FROM teacher_comments WHERE registration_no = ? AND class = ? AND term = ? AND year = ?", 
    [$registration_no, $myclass, $term, $examyear], "rows")[0] ?? [];

$teacher_general_comment = $overall_teacher_comments['comment'] ?? '';
$principal_comment = $overall_teacher_comments['principal_comment'] ?? '';

// Get subject teachers
$subject_teachers = [];
foreach ($comments as $comment) {
    if (!empty($comment['teacher_name'])) {
        $subject_teachers[$comment['subject']] = $comment['teacher_name'];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Verbum Dei Academy - Pre-Nursery Report</title>
    <!-- Bootstrap 5 CDN -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome CDN -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;600;700&family=Playfair+Display:wght@700&family=Short+Stack&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-color: #4CACC0;
            --secondary-color: #81C784;
            --accent-color: #C8E6C9;
            --dark-color: #388E3C;
            --light-color: #E8F5E9;
            --success-color: #66BB6A;
            --warning-color: #FFB74D;
            --prenur-pink: #F48FB1;
            --prenur-light-pink: #FCE4EC;
            --prenur-blue: #90CAF9;
            --prenur-yellow: #FFF59D;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: #f8f9fa;
            color: #333;
        }
        
        .header {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--prenur-blue) 100%);
            color: white;
            padding: 1.5rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            border-bottom: 8px solid var(--prenur-yellow);
        }
        
        .school-name {
            font-family: 'Playfair Display', serif;
            font-weight: 700;
            font-size: 2.2rem;
            margin-bottom: 0.5rem;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
        }
        
        .school-motto {
            font-style: italic;
            opacity: 0.9;
            font-size: 0.9rem;
        }
        
        .prenur-badge {
            background: linear-gradient(90deg, var(--prenur-pink) 0%, var(--prenur-yellow) 100%);
            color: #2C3E50;
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 700;
            letter-spacing: 0.5px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 6px 15px rgba(0,0,0,0.08);
            margin-bottom: 1.5rem;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            border-top: 5px solid var(--prenur-blue);
            overflow: hidden;
        }
        
        .card:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.12);
        }
        
        .card-header {
            background: linear-gradient(135deg, var(--accent-color) 0%, var(--secondary-color) 100%);
            color: #2C3E50;
            font-weight: 700;
            border-radius: 15px 15px 0 0 !important;
            padding: 1rem 1.5rem;
            border-bottom: 3px solid var(--prenur-yellow);
            font-size: 1.1rem;
        }
        
        .profile-pic {
            border: 5px solid white;
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
            transition: transform 0.3s ease;
        }
        
        .profile-pic:hover {
            transform: scale(1.05);
        }
        
        .table {
            border-radius: 10px;
            overflow: hidden;
            border: 1px solid #E0E0E0;
        }
        
        .table th {
            background-color: var(--light-color);
            color: var(--dark-color);
            font-weight: 700;
            border-bottom: 3px solid var(--prenur-yellow);
            padding: 1rem;
            font-size: 0.95rem;
        }
        
        .table td {
            padding: 1rem;
            vertical-align: top;
        }
        
        .subject-row {
            background-color: white;
            transition: all 0.3s ease;
            border-left: 4px solid var(--prenur-blue);
        }
        
        .subject-row:hover {
            background-color: var(--prenur-light-pink);
            transform: translateX(5px);
        }
        
        .subject-name {
            font-weight: 600;
            color: var(--dark-color);
            font-size: 1.1rem;
            margin-bottom: 5px;
        }
        
        .teacher-name {
            font-size: 0.85rem;
            color: #666;
            font-style: italic;
        }
        
        .comment-box {
            background-color: var(--light-color);
            border-left: 4px solid var(--prenur-pink);
            padding: 15px;
            border-radius: 8px;
            margin-top: 10px;
            font-family: 'Poppins', sans-serif;
            line-height: 1.6;
            white-space: pre-line;
        }
        
        .comment-box p {
            margin-bottom: 10px;
        }
        
        .btn-print {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            color: white;
            font-weight: 700;
            padding: 0.75rem 2rem;
            border-radius: 10px;
            border: none;
            transition: all 0.3s ease;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        
        .btn-print:hover {
            background: linear-gradient(135deg, var(--dark-color) 0%, var(--primary-color) 100%);
            color: white;
            transform: translateY(-3px);
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
        
        .btn-back {
            background: linear-gradient(135deg, var(--prenur-blue) 0%, #64B5F6 100%);
            color: white;
            font-weight: 700;
            padding: 0.75rem 2rem;
            border-radius: 10px;
            border: none;
            transition: all 0.3s ease;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        
        .btn-back:hover {
            background: linear-gradient(135deg, #2196F3 0%, var(--prenur-blue) 100%);
            color: white;
            transform: translateY(-3px);
        }
        
        .footer {
            background: linear-gradient(135deg, var(--dark-color) 0%, var(--primary-color) 100%);
            color: white;
            padding: 1.5rem 0;
            margin-top: 3rem;
            border-top: 5px solid var(--prenur-yellow);
        }
        
        .attendance-circle {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            background: conic-gradient(var(--success-color) 0% calc(var(--percentage) * 1%), #e0e0e0 calc(var(--percentage) * 1%) 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto;
            position: relative;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        
        .attendance-inner {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 1.2rem;
            color: var(--dark-color);
        }
        
        .bubble {
            background-color: var(--prenur-light-pink);
            border-radius: 20px;
            padding: 15px 20px;
            margin: 10px 0;
            position: relative;
            border: 2px solid var(--prenur-pink);
        }
        
        .bubble:before {
            content: '';
            position: absolute;
            top: -10px;
            left: 30px;
            border-left: 10px solid transparent;
            border-right: 10px solid transparent;
            border-bottom: 10px solid var(--prenur-pink);
        }
        
        .teacher-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--prenur-blue) 0%, var(--prenur-pink) 100%);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            font-weight: 700;
            margin-right: 15px;
        }
        
        .subject-icon {
            color: var(--prenur-pink);
            font-size: 1.2rem;
            margin-right: 10px;
        }
        
        /* Print Styles */
        @media print {
            body {
                background-color: white;
                font-size: 10pt;
                padding: 0;
                margin: 0;
            }
            
            .no-print {
                display: none !important;
            }
            
            .header {
                background: white !important;
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
                color: #2C3E50 !important;
                border-bottom: 3px solid #2C3E50 !important;
            }
            
            .card {
                box-shadow: none !important;
                border: 1px solid #ddd !important;
                page-break-inside: avoid !important;
            }
            
            .card-header {
                background: #f0f0f0 !important;
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
                color: #2C3E50 !important;
            }
            
            .btn-print, .btn-back {
                display: none !important;
            }
            
            .footer {
                background: #2C3E50 !important;
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }
            
            @page {
                size: A4;
                margin: 1.5cm;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header text-center">
        <div class="container">
            <div class="mb-3">
                <img src="vda_logo_pri.png" alt="Verbum Dei Academy" height="90" class="me-3">
            </div>
            <h1 class="school-name">Verbum Dei Academy</h1>
            <h4 class="mb-3">Pre-Nursery & Reception Report Card</h4>
            <span class="prenur-badge">
                <i class="fas fa-child me-1"></i> Early Childhood Education
            </span>
        </div>
    </div>

    <!-- Main Content -->
    <div class="container">
        <?php if ($msg) echo $msg; ?>
        
        <!-- Student Info -->
        <div class="card">
            <div class="card-header">
                <i class="fas fa-user-graduate me-2"></i> Student Information
            </div>
            <div class="card-body">
                <div class="row align-items-center">
                    <div class="col-md-2 text-center">
                        <?php echo $profile_pic; ?>
                        <div class="mt-3">
                            <span class="badge rounded-pill" style="background-color: var(--prenur-pink); color: white; padding: 8px 15px;">
                                <i class="fas fa-school me-1"></i> <?php echo htmlspecialchars($myclass); ?>
                            </span>
                        </div>
                    </div>
                    <div class="col-md-10">
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-user me-3" style="color: var(--prenur-pink); font-size: 1.2rem;"></i>
                                    <div>
                                        <div class="text-muted small">Student Name</div>
                                        <div class="fw-bold fs-5"><?php echo htmlspecialchars($user_name); ?></div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-id-card me-3" style="color: var(--prenur-blue); font-size: 1.2rem;"></i>
                                    <div>
                                        <div class="text-muted small">Registration No</div>
                                        <div class="fw-bold"><?php echo htmlspecialchars($registration_no); ?></div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-venus-mars me-3" style="color: var(--prenur-pink); font-size: 1.2rem;"></i>
                                    <div>
                                        <div class="text-muted small">Gender</div>
                                        <div class="fw-bold"><?php echo htmlspecialchars($user_gender); ?></div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-child me-3" style="color: var(--prenur-blue); font-size: 1.2rem;"></i>
                                    <div>
                                        <div class="text-muted small">Age</div>
                                        <div class="fw-bold"><?php echo $age; ?> Years</div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-chalkboard-teacher me-3" style="color: var(--prenur-pink); font-size: 1.2rem;"></i>
                                    <div>
                                        <div class="text-muted small">Class Teacher</div>
                                        <div class="fw-bold"><?php echo htmlspecialchars($teacher_name); ?></div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-calendar me-3" style="color: var(--prenur-blue); font-size: 1.2rem;"></i>
                                    <div>
                                        <div class="text-muted small">Academic Session</div>
                                        <div class="fw-bold"><?php echo htmlspecialchars($examyear); ?> - <?php echo htmlspecialchars($term); ?></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Attendance -->
        <div class="card">
            <div class="card-header">
                <i class="fas fa-calendar-check me-2"></i> Attendance Record
            </div>
            <div class="card-body">
                <div class="row align-items-center">
                    <div class="col-md-4 text-center">
                        <div class="attendance-circle" style="--percentage: <?php echo $SchOpen > 0 ? ($Present/$SchOpen)*100 : 0; ?>;">
                            <div class="attendance-inner">
                                <?php echo $SchOpen > 0 ? round(($Present/$SchOpen)*100, 0) : 0; ?>%
                            </div>
                        </div>
                        <div class="mt-2 text-muted">Attendance Rate</div>
                    </div>
                    <div class="col-md-8">
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <div class="text-center p-3 rounded" style="background-color: var(--light-color);">
                                    <div class="fw-bold fs-4" style="color: var(--primary-color);"><?php echo $SchOpen; ?></div>
                                    <div class="text-muted small">Total School Days</div>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="text-center p-3 rounded" style="background-color: var(--prenur-light-pink);">
                                    <div class="fw-bold fs-4" style="color: var(--success-color);"><?php echo $Present; ?></div>
                                    <div class="text-muted small">Days Present</div>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="text-center p-3 rounded" style="background-color: #FFEBEE;">
                                    <div class="fw-bold fs-4" style="color: #F44336;"><?php echo $Absent; ?></div>
                                    <div class="text-muted small">Days Absent</div>
                                </div>
                            </div>
                        </div>
                        <div class="row mt-3">
                            <div class="col-md-6">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-calendar-day me-3" style="color: var(--prenur-blue);"></i>
                                    <div>
                                        <div class="text-muted small">Term Ends</div>
                                        <div class="fw-bold"><?php echo htmlspecialchars($DateOfReport); ?></div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-calendar-plus me-3" style="color: var(--prenur-pink);"></i>
                                    <div>
                                        <div class="text-muted small">Next Term Begins</div>
                                        <div class="fw-bold"><?php echo htmlspecialchars($NextTermBegins); ?></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Subject Comments -->
        <div class="card">
            <div class="card-header">
                <i class="fas fa-comment-alt me-2"></i> Subject Teachers' Comments
            </div>
            <div class="card-body">
                <?php if (empty($comments)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-comments fa-4x mb-3" style="color: var(--prenur-blue); opacity: 0.5;"></i>
                        <h5 class="text-muted mb-2">No Subject Comments Available</h5>
                        <p class="text-muted">Subject teachers' comments have not been uploaded for this term.</p>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th width="25%">Subject</th>
                                    <th width="75%">Teacher's Comments</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($comments as $comment): ?>
                                    <tr class="subject-row">
                                        <td>
                                            <div class="subject-name">
                                                <i class="fas fa-book subject-icon"></i>
                                                <?php echo htmlspecialchars($comment['subject']); ?>
                                            </div>
                                            <?php if (!empty($comment['teacher_name'])): ?>
                                                <div class="teacher-name">
                                                    <i class="fas fa-chalkboard-teacher me-1"></i>
                                                    <?php echo htmlspecialchars($comment['teacher_name']); ?>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="comment-box">
                                                <?php 
                                                $formatted_comment = htmlspecialchars($comment['comment']);
                                                // Replace newlines with line breaks
                                                $formatted_comment = nl2br($formatted_comment);
                                                // Add some basic formatting
                                                $formatted_comment = str_replace('Additional:', '<strong>Additional:</strong>', $formatted_comment);
                                                $formatted_comment = str_replace('Progressive Report:', '<strong>Progressive Report:</strong>', $formatted_comment);
                                                echo $formatted_comment; 
                                                ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Subject Teachers Summary -->
                    <div class="mt-4 p-3 rounded" style="background-color: var(--prenur-light-pink); border-left: 4px solid var(--prenur-pink);">
                        <h6 class="mb-3" style="color: var(--dark-color);">
                            <i class="fas fa-chalkboard-teacher me-2"></i> Subject Teachers Summary
                        </h6>
                        <div class="row">
                            <?php 
                            $unique_teachers = array_unique(array_column($comments, 'teacher_name'));
                            foreach ($unique_teachers as $teacher): 
                                if (!empty($teacher)):
                                    $teacher_subjects = array_filter($comments, function($c) use ($teacher) {
                                        return $c['teacher_name'] == $teacher;
                                    });
                                    $subject_count = count($teacher_subjects);
                            ?>
                                <div class="col-md-4 mb-2">
                                    <div class="d-flex align-items-center">
                                        <div class="teacher-avatar">
                                            <?php echo substr($teacher, 0, 1); ?>
                                        </div>
                                        <div>
                                            <div class="fw-bold"><?php echo htmlspecialchars($teacher); ?></div>
                                            <div class="text-muted small">
                                                <?php echo $subject_count; ?> subject<?php echo $subject_count > 1 ? 's' : ''; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php 
                                endif;
                            endforeach; 
                            ?>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- General Comments -->
        <div class="card">
            <div class="card-header">
                <i class="fas fa-comments me-2"></i> General Comments & Recommendations
            </div>
            <div class="card-body">
                <!-- Class Teacher's Remarks -->
                <div class="mb-4">
                    <div class="d-flex align-items-center mb-3">
                        <div class="teacher-avatar">
                            <?php echo substr($teacher_name, 0, 1); ?>
                        </div>
                        <div>
                            <h5 class="mb-1" style="color: var(--dark-color);">
                                <i class="fas fa-chalkboard-teacher me-2"></i> Class Teacher's Remarks
                            </h5>
                            <div class="text-muted small"><?php echo htmlspecialchars($teacher_name); ?></div>
                        </div>
                    </div>
                    
                    <?php if (!empty($teacher_general_comment)): ?>
                        <div class="bubble">
                            <?php echo nl2br(htmlspecialchars($teacher_general_comment)); ?>
                        </div>
                    <?php elseif (!empty($TeachersRemarks)): ?>
                        <div class="bubble">
                            <?php echo nl2br(htmlspecialchars($TeachersRemarks)); ?>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-4 text-muted">
                            <i class="fas fa-comment-slash fa-2x mb-2"></i>
                            <p>No general remarks from class teacher</p>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Head of School's Comments -->
                <div class="mb-4">
                    <div class="d-flex align-items-center mb-3">
                        <div class="teacher-avatar" style="background: linear-gradient(135deg, #FF9800 0%, #FF5722 100%);">
                            H
                        </div>
                        <div>
                            <h5 class="mb-1" style="color: var(--dark-color);">
                                <i class="fas fa-user-tie me-2"></i> Head of School's Comments
                            </h5>
                            <div class="text-muted small">Head of School</div>
                        </div>
                    </div>
                    
                    <?php if (!empty($principal_comment)): ?>
                        <div class="bubble">
                            <?php echo nl2br(htmlspecialchars($principal_comment)); ?>
                        </div>
                    <?php elseif (!empty($PrincipalsComment)): ?>
                        <div class="bubble">
                            <?php echo nl2br(htmlspecialchars($PrincipalsComment)); ?>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-4 text-muted">
                            <i class="fas fa-comment-slash fa-2x mb-2"></i>
                            <p>No comments from Head of School</p>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Term Summary -->
                <div class="mt-5">
                    <div class="row">
                        <div class="col-md-4">
                            <div class="text-center p-3 rounded" style="background-color: var(--light-color);">
                                <div class="fw-bold fs-3" style="color: var(--primary-color);">
                                    <?php echo count($comments); ?>
                                </div>
                                <div class="text-muted small">Subjects with Comments</div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="text-center p-3 rounded" style="background-color: var(--prenur-light-pink);">
                                <div class="fw-bold fs-3" style="color: var(--prenur-pink);">
                                    <?php echo $Present; ?>/<?php echo $SchOpen; ?>
                                </div>
                                <div class="text-muted small">Days Attended</div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="text-center p-3 rounded" style="background-color: #E3F2FD;">
                                <div class="fw-bold fs-3" style="color: var(--prenur-blue);">
                                    <?php echo count(array_unique(array_column($comments, 'teacher_name'))); ?>
                                </div>
                                <div class="text-muted small">Subject Teachers</div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Signature Section -->
                <div class="text-center mt-5 pt-4 border-top">
                    <div class="row justify-content-center">
                        <div class="col-md-4 mb-4">
                            <div class="p-3">
                                <div class="mb-3"><strong>Class Teacher's Signature</strong></div>
                                <div class="signature-line" style="border-bottom: 2px solid #666; width: 200px; margin: 0 auto;"></div>
                                <div class="mt-2 text-muted"><?php echo htmlspecialchars($teacher_name); ?></div>
                            </div>
                        </div>
                        <div class="col-md-4 mb-4">
                            <div class="p-3">
                                <div class="mb-3"><strong>Head of School's Signature</strong></div>
                                <div class="signature-line" style="border-bottom: 2px solid #666; width: 200px; margin: 0 auto;"></div>
                                <div class="mt-2 text-muted">Head of School</div>
                            </div>
                        </div>
                        <div class="col-md-4 mb-4">
                            <div class="p-3">
                                <div class="mb-3"><strong>School Stamp</strong></div>
                                <img src="stamp.png" alt="School Stamp" width="120" height="auto" style="opacity: 0.9;">
                            </div>
                        </div>
                    </div>
                    <div class="alert alert-info mt-3" style="background-color: var(--accent-color); border-color: var(--secondary-color);">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Note:</strong> This report focuses on developmental progress and qualitative assessment. 
                        Pre-Nursery & Reception reports emphasize observations, skills development, and overall growth rather than numerical grades.
                    </div>
                </div>
            </div>
        </div>

        <!-- Print Button -->
        <div class="text-center no-print my-5">
            <div class="d-flex justify-content-center gap-3">
                <a href="main.php" class="btn btn-back">
                    <i class="fas fa-arrow-left me-2"></i> Back to Dashboard
                </a>
                <button onclick="window.print()" class="btn btn-print">
                    <i class="fas fa-print me-2"></i> Print Report Card
                </button>
            </div>
            
            <div class="mt-4">
                <div class="alert alert-warning d-inline-block" style="background-color: var(--prenur-yellow); border-color: #FFD54F;">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Important:</strong> Keep this report card safe. It will be required for next term's registration.
                </div>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <footer class="footer text-center">
        <div class="container">
            <div class="mb-3">
                <img src="vda_logo_pri.png" alt="Verbum Dei Academy" height="50" class="me-3">
            </div>
            <p class="mb-0">
                &copy; <?php echo date('Y'); ?> Verbum Dei Academy. All Rights Reserved.<br>
                <small>Early Childhood Education Report System</small>
            </p>
            <div class="mt-2">
                <span class="badge rounded-pill" style="background-color: rgba(255,255,255,0.2); padding: 5px 15px;">
                    <i class="fas fa-child me-1"></i> Pre-Nursery & Reception
                </span>
            </div>
        </div>
    </footer>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
    // Enhance print functionality
    document.addEventListener('DOMContentLoaded', function() {
        // Add print-specific styling
        const printStyles = `
            @media print {
                .no-print { display: none !important; }
                .card { break-inside: avoid; }
                .table { font-size: 9pt; }
                .comment-box { font-size: 9pt; }
                .bubble { border: 1px solid #ccc; }
                .attendance-circle { -webkit-print-color-adjust: exact; }
            }
        `;
        
        const styleSheet = document.createElement("style");
        styleSheet.type = "text/css";
        styleSheet.innerText = printStyles;
        document.head.appendChild(styleSheet);
        
        // Print button enhancement
        const printBtn = document.querySelector('.btn-print');
        if (printBtn) {
            printBtn.addEventListener('click', function(e) {
                e.preventDefault();
                window.print();
            });
        }
        
        // Add animation to subject rows on hover
        const subjectRows = document.querySelectorAll('.subject-row');
        subjectRows.forEach(row => {
            row.addEventListener('mouseenter', function() {
                this.style.transition = 'all 0.3s ease';
            });
        });
    });
    </script>
</body>
</html>